!--------------------------------------------------------------------
subroutine mdeqcor1(de,pr,v1,v2,v3,pg,is1,ie1,js1,je1,ks1,ke1)
!--------------------------------------------------------------------
!  Hydrostatic Corona Model 
!
  use pram, only : imax, jmax, kmax, gam, c0, rbh, metric
  implicit none

  integer :: i, j, k, is1 ,ie1, js1, je1, ks1, ke1

  real(8) :: de(is1:ie1,js1:je1,ks1:ke1), pr(is1:ie1,js1:je1,ks1:ke1), &
             v1(is1:ie1,js1:je1,ks1:ke1), v2(is1:ie1,js1:je1,ks1:ke1), &
             v3(is1:ie1,js1:je1,ks1:ke1)
  real(8) :: pg(is1:ie1,js1:je1,ks1:ke1)

  real(8) :: gcm, dd0, ee0, rd, cssq, pg0, consr, alpg, anpg

!--------------------------------------------------------------------
! Parameter
!
  gcm=0.5*rbh

  cssq=gam*(gam-1.0)*ee0/dd0
         
  pg0=-gcm/abs(rd)
  consr=(c0*c0+cssq/(gam-1.0))*gam/cssq*0.5
  alpg=log(1.0+2.0*pg0/c0**2)

!--------------------------------------------------------------------
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1  

        if(metric .gt. 100 .and. abs(alpg).gt.0.d0) then
          anpg=2.0*pg(i,j,k)/c0**2
          de(i,j,k)=dd0*exp(-consr*(log(1.0+anpg)-alpg))
        else
          de(i,j,k)=dd0*exp(-gam*(pg(i,j,k)-pg0)/cssq)
        endif

        pr(i,j,k)=cssq/gam*de(i,j,k)
!
        v1(i,j,k)=0.d0
        v2(i,j,k)=0.d0
        v3(i,j,k)=0.d0
!
      enddo
    enddo
  enddo

  return
end subroutine mdeqcor1
!
!--------------------------------------------------------------------
subroutine mdeqcor2(de,pr,v1,v2,v3,x1,is1,ie1,js1,je1,ks1,ke1,dd0,ee0)
!--------------------------------------------------------------------
!  Hydrostatic Corona Model (Komissarov Model)
!
  use pram, only : imax, jmax, kmax, gam, c0
  implicit none

  integer :: i, j, k, is1 ,ie1, js1, je1, ks1, ke1

  real(8) :: de(is1:ie1,js1:je1,ks1:ke1), pr(is1:ie1,js1:je1,ks1:ke1), &
             v1(is1:ie1,js1:je1,ks1:ke1), v2(is1:ie1,js1:je1,ks1:ke1), &
             v3(is1:ie1,js1:je1,ks1:ke1)
  real(8) :: x1(imax)
!
  real(8) :: dd0, ee0, rdcen1, ck1, cssq, rr
!
!--------------------------------------------------------------------
! Parameter for Hydrostatic Corona
!
  rdcen1=7.5d0
  ck1=0.01d0

  cssq=gam*(gam-1.0)*ee0/dd0
!
!--------------------------------------------------------------------
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1  

        rr=x1(i)
        de(i,j,k)=0.1*dd0*exp(-3.0*rr/rdcen1)

!        pr(i,j,k)=ck1*de(i,j,k)**gam
        pr(i,j,k)=ck1*de(i,j,k)**((gam+1.0)/gam)
!        pr(i,j,k)=cssq/gam*de(i,j,k)
!        pr(i,j,k)=(gam-1.0)*e0*de(i,j,k)

        v1(i,j,k)=0.d0
        v2(i,j,k)=0.d0
        v3(i,j,k)=0.d0

      enddo
    enddo
  enddo
!
  return
end subroutine mdeqcor2
!
!--------------------------------------------------------------------
subroutine mdeqcor3(de,pr,v1,v2,v3,x1,is1,ie1,js1,je1,ks1,ke1,dd0)
!--------------------------------------------------------------------
!  Hydrostatic Corona Model (McKinney Model)
!
  use pram, only : imax, jmax, kmax, gam, c0
  implicit none

  integer :: i, j, k, is1 ,ie1, js1, je1, ks1, ke1

  real(8) :: de(is1:ie1,js1:je1,ks1:ke1), pr(is1:ie1,js1:je1,ks1:ke1), &
             v1(is1:ie1,js1:je1,ks1:ke1), v2(is1:ie1,js1:je1,ks1:ke1), &
             v3(is1:ie1,js1:je1,ks1:ke1)
  real(8) :: x1(imax)
!
  real(8) :: dd0, rr, uutmp1
!
!--------------------------------------------------------------------
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1  

        rr=x1(i)
          
        de(i,j,k)=0.1*dd0*rr**(-3.0/2.0)
           
        uutmp1=0.01*dd0*rr**(-5.0/2.0)
        pr(i,j,k)=(gam-1.0)*de(i,j,k)*uutmp1

        v1(i,j,k)=0.d0
        v2(i,j,k)=0.d0
        v3(i,j,k)=0.d0

      enddo
    enddo
  enddo
!
  return
end subroutine mdeqcor3
!
!--------------------------------------------------------------------
subroutine mdcor1(de,pr,v1,v2,v3,is1,ie1,js1,je1,ks1,ke1)
!--------------------------------------------------------------------
!  Constant density, pressure (for 1D GR Test)
!
  use pram, only : imax, jmax, kmax, gam, c0
  implicit none

  integer :: i, j, k, is1 ,ie1, js1, je1, ks1, ke1

  real(8) :: de(is1:ie1,js1:je1,ks1:ke1), pr(is1:ie1,js1:je1,ks1:ke1), &
             v1(is1:ie1,js1:je1,ks1:ke1), v2(is1:ie1,js1:je1,ks1:ke1), &
             v3(is1:ie1,js1:je1,ks1:ke1)
!
!--------------------------------------------------------------------
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1  

        de(i,j,k)=1.d0
        pr(i,j,k)=0.1d0
        v1(i,j,k)=0.d0
        v2(i,j,k)=0.d0
        v3(i,j,k)=0.d0

      enddo
    enddo
  enddo
!
  return
end subroutine mdcor1
!
!--------------------------------------------------------------------
subroutine mdffcor(de,pr,v1,v2,v3,x1,is1,ie1,js1,je1,ks1,ke1,&
                   dd0,ee0,vp,rd,hrel,rc,rshock)
!--------------------------------------------------------------------
!   Bondi Accretion Model
!
  use pram, only : imax, jmax, kmax, gam, c0, rbh, xmax
  implicit none

  integer :: i, j, k, is1 ,ie1, js1, je1, ks1, ke1
  integer :: merr

  real(8) :: de(is1:ie1,js1:je1,ks1:ke1), pr(is1:ie1,js1:je1,ks1:ke1), &
             v1(is1:ie1,js1:je1,ks1:ke1), v2(is1:ie1,js1:je1,ks1:ke1), &
             v3(is1:ie1,js1:je1,ks1:ke1)
  real(8) :: x1(imax)
  real(8), allocatable :: u(:), uw(:)

  real(8) :: dd0, ee0, vp, rd, gam0, hrel, rc, rshock, &
             vsmall, ard, xxp, fxp, ushk, al, gl, amon, fuw, xshock, uflow
!
  allocate(u(imax), uw(imax), stat=merr)
! 
!--------------------------------------------------------------------
!  Parameter
!
  vsmall=1.d-10
  ard=abs(rd)
!
!--------------------------------------------------------------------
!  Transonic Solution for Free-Fall Corona
!
  if( rc .lt. 0.d0 .and. ee0 .gt. vsmall) then
    call caxpoi(xxp,fxp,hrel)
!
!    if( rshock .gt. 0.d0 .and. rshock .lt. xmax) then
!      call washk1(rshock,ushk,xxp,fxp,hrel)
!      al=sqrt(1.0-rbh/rshock)
!      gl=sqrt(ushk**2+1.0)
!      amon=(hrel/al*ushk**2/gl+(gam-1.0)/gam*(hrel/al/gl-1.0))/ushk
!      call cupwin(fuw,rshock,amon,hrel)
!    endif
!
    call washk1(x1,u,xxp,fxp,hrel)
!
    if( rshock .gt. 0.d0 .and. rshock .lt. xmax) then
      xshock=rshock/rbh
      call washk2(x1,uw,xshock,fuw,hrel)
    endif 
  endif
!
!--------------------------------------------------------------------
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1  
!  Pressure-less free fall
        if( ee0 .eq. 0.d0 ) then
          de(i,j,k)=0.1*dd0*abs(x1(i)/ard)**(-1.5)
          pr(i,j,k)=(gam-1.0)*ee0*(de(i,j,k)/dd0)**gam
           
          v1(i,j,k)=-c0*vp*sqrt(rbh/x1(i))
          v2(i,j,k)=0.d0
          v3(i,j,k)=0.d0
!
!  Transonic solution

        else
          if( rshock .le. 0.d0 ) then
            uflow=u(i)
          else
            if( x1(i) .le. rshock ) then
              uflow=u(i)
            else
              uflow=uw(i)
            endif
          endif  
         
          al=sqrt(1.0-rbh/x1(i))
          de(i,j,k)=dd0/uflow/al/x1(i)**2
           
          gl=sqrt(uflow**2+1.0)
          pr(i,j,k)=(gam-1.0)/gam*(hrel/al/gl-1.0)*de(i,j,k)*c0**2

          v1(i,j,k)=-c0*vp*uflow/gl
          v2(i,j,k)=0.d0
          v3(i,j,k)=0.d0
        endif
    
      enddo
    enddo
  enddo

  deallocate(u,uw, stat=merr)
!
  return
end subroutine mdffcor
!
!--------------------------------------------------------------------
subroutine mdffcor1(de,pr,v1,v2,v3,x1,x3,hh,is1,ie1,js1,je1,ks1,ke1,&
                   dd0,vp,rc)
!--------------------------------------------------------------------
!   Bondi Accretion Model based on Hawley, Smarr, Wilson (1984)
!
  use pram, only : imax, jmax, kmax, gam, c0, rbh, metric, akm
  implicit none

  integer :: i, j, k, is1 ,ie1, js1, je1, ks1, ke1
  integer :: merr, iflg1, irs

  real(8) :: de(is1:ie1,js1:je1,ks1:ke1), pr(is1:ie1,js1:je1,ks1:ke1), &
             v1(is1:ie1,js1:je1,ks1:ke1), v2(is1:ie1,js1:je1,ks1:ke1), &
             v3(is1:ie1,js1:je1,ks1:ke1)

  real(8) :: hh(0:3,is1:ie1,js1:je1,ks1:ke1)

  real(8) :: x1(imax), x3(kmax)

  real(8), allocatable :: ffur(:), ffte(:)

  real(8) :: dd0, vp, rc, gam1, urcsq, urc, vvcsq, tec, cnst1, cnst2, &
             rr, qt, ur, te, ut, gf, gtt, grr, alp1, rg, arg, del, sig, alp   
!
  allocate(ffur(imax), ffte(imax), stat=merr)
!
! 
!--------------------------------------------------------------------
!  Parameter
!
  gam1=1.d0/(gam-1.d0)
!
  urcsq=(0.5*rbh)/(2.*rc)
  urc=-sqrt(urcsq)
  vvcsq=(urcsq)/(1.-3.*urcsq)
  tec=gam1*vvcsq/((1.+gam1)*(1.-gam1*vvcsq))
! 
  cnst1=(tec**gam1)*urc*(rc**2)
  cnst2=(1.-(rbh/rc)+urcsq)*(1.+(1.+gam1)*tec)**2
!
  iflg1=0
!      
  do i=1,imax
    if(iflg1 .eq. 0) then
      if(x1(i) .ge. rc) then
        iflg1=1
        irs=i
      endif
    endif
  enddo
!
!======================================================================@
!     Calculation of alpha and ur by Newton-Raphson
!======================================================================@
!
  do i=1,imax
    ffur(i)=urc
    ffte(i)=tec
  enddo
!
! ==== rr > rcs ====  
!
  do i=irs,imax,1
  
    rr=x1(i)
    
    if(i .gt. irs) then
      ur=ffur(i-1)+0.01
      te=ffte(i-1)
    else
      ur=urc
      te=tec
    endif
       
    call calws1(rr,ur,te,cnst1,cnst2)
    
    ffur(i)=ur
    ffte(i)=te

    
  enddo
!
! ==== rr < rcs ====
!
  do i=irs-1,1,-1

    rr=x1(i)

    if(i .eq. irs-1) then
      ur=urc
      te=tec
    else
      ur=ffur(i+1)-0.01
      te=ffte(i+1)
    endif

    call calws1(rr,ur,te,cnst1,cnst2)

    ffur(i)=ur
    ffte(i)=te

  enddo
    
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1 
 
        rr=x1(i)
        qt=x3(k)

        if(metric .eq. 103) then
          gtt=-1.0*hh(0,i,j,k)**2
          grr=hh(1,i,j,k)**2
          alp1=hh(0,i,j,k)
        elseif(metric .eq. 203) then
          rg=0.5*rbh
          arg=akm*rg
          del=rr**2.0-2.0*rg*rr+arg**2.0
          sig=rr**2.0+(arg*cos(qt))**2.0
          alp=(rr**2.0+arg**2.0)**2.0-del*(arg*sin(qt))**2.0
!
          gtt=-(1.0-(2.0*rg*rr/sig))
          grr=hh(1,i,j,k)**2
          alp1=sqrt(sig*del/alp)
!
        endif
!
        ut=sqrt((-1.-grr*ffur(i)**2)/gtt)
        gf=alp1*ut
!
        de(i,j,k)=dd0*ffte(i)**gam
        pr(i,j,k)=de(i,j,k)*ffte(i)
        v1(i,j,k)=vp*hh(1,i,j,k)*ffur(i)/gf
        v2(i,j,k)=0.d0
        v3(i,j,k)=0.d0          
!
      enddo
    enddo
  enddo
!
  deallocate(ffur,ffte, stat=merr)
 
  return
end subroutine mdffcor1
!
!--------------------------------------------------------------------
subroutine mdffcor2(de,pr,v1,v2,v3,x1,x3,hh,is1,ie1,js1,je1,ks1,ke1,&
                   dd0,vp,rc,al0)
!--------------------------------------------------------------------
!   Bondi Accretion Model based on Hawley, Smarr, Wilson (1984)
!   + angular momentum (Proga et al. 2003)
!
  use pram, only : imax, jmax, kmax, gam, c0, rbh, metric, akm, pi
  implicit none

  integer :: i, j, k, is1 ,ie1, js1, je1, ks1, ke1
  integer :: merr, iflg1, irs

  real(8) :: de(is1:ie1,js1:je1,ks1:ke1), pr(is1:ie1,js1:je1,ks1:ke1), &
             v1(is1:ie1,js1:je1,ks1:ke1), v2(is1:ie1,js1:je1,ks1:ke1), &
             v3(is1:ie1,js1:je1,ks1:ke1)

  real(8) :: hh(0:3,is1:ie1,js1:je1,ks1:ke1)

  real(8) :: x1(imax), x3(kmax)

  real(8), allocatable :: ffur(:), ffte(:)

  real(8) :: dd0, vp, rc, gam1, urcsq, urc, vvcsq, tec, cnst1, cnst2, &
             rr, qt, ur, te, ut, gf, gtt, grr, alp1, rg, arg, del, sig, alp, &
             al0, ffunc, uphi   
!
  allocate(ffur(imax), ffte(imax), stat=merr)
!
! 
!--------------------------------------------------------------------
!  Parameter
!
  gam1=1.d0/(gam-1.d0)
!
  urcsq=(0.5*rbh)/(2.*rc)
  urc=-sqrt(urcsq)
  vvcsq=(urcsq)/(1.-3.*urcsq)
  tec=gam1*vvcsq/((1.+gam1)*(1.-gam1*vvcsq))
! 
  cnst1=(tec**gam1)*urc*(rc**2)
  cnst2=(1.-(rbh/rc)+urcsq)*(1.+(1.+gam1)*tec)**2
!
  iflg1=0
!      
  do i=1,imax
    if(iflg1 .eq. 0) then
      if(x1(i) .ge. rc) then
        iflg1=1
        irs=i
      endif
    endif
  enddo
!
!======================================================================@
!     Calculation of alpha and ur by Newton-Raphson
!======================================================================@
!
  do i=1,imax
    ffur(i)=urc
    ffte(i)=tec
  enddo
!
! ==== rr > rcs ====  
!
  do i=irs,imax,1
  
    rr=x1(i)
    
    if(i .gt. irs) then
      ur=ffur(i-1)+0.01
      te=ffte(i-1)
    else
      ur=urc
      te=tec
    endif
       
    call calws1(rr,ur,te,cnst1,cnst2)
    
    ffur(i)=ur
    ffte(i)=te

    
  enddo
!
! ==== rr < rcs ====
!
  do i=irs-1,1,-1

    rr=x1(i)

    if(i .eq. irs-1) then
      ur=urc
      te=tec
    else
      ur=ffur(i+1)-0.01
      te=ffte(i+1)
    endif

    call calws1(rr,ur,te,cnst1,cnst2)

    ffur(i)=ur
    ffte(i)=te

  enddo
    
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1 
 
        rr=x1(i)
        qt=x3(k)

        if(metric .eq. 103) then
          gtt=-1.0*hh(0,i,j,k)**2
          grr=hh(1,i,j,k)**2
          alp1=hh(0,i,j,k)
        elseif(metric .eq. 203) then
          rg=0.5*rbh
          arg=akm*rg
          del=rr**2.0-2.0*rg*rr+arg**2.0
          sig=rr**2.0+(arg*cos(qt))**2.0
          alp=(rr**2.0+arg**2.0)**2.0-del*(arg*sin(qt))**2.0
!
          gtt=-(1.0-(2.0*rg*rr/sig))
          grr=hh(1,i,j,k)**2
          alp1=sqrt(sig*del/alp)
!
        endif
!
        ut=sqrt((-1.-grr*ffur(i)**2)/gtt)
        gf=alp1*ut
!
        de(i,j,k)=dd0*ffte(i)**gam
        pr(i,j,k)=de(i,j,k)*ffte(i)
        v1(i,j,k)=vp*hh(1,i,j,k)*ffur(i)/gf
        v2(i,j,k)=0.d0
        v3(i,j,k)=0.d0          
!
!  Angular momentum
!
        ffunc=1.- abs(cos(qt))
!        ffunc=1.- (cos(qt)**10.0)
!        if(qt .le. theta0 .or. qt .ge. pi-theta0)
!          ffunc=0.d0
!        else
!          ffunc=1.d0
!        endif
!
        uphi=al0*ffunc/rr 
        v2(i,j,k)=hh(2,i,j,k)*uphi/gf                
!
      enddo
    enddo
  enddo
!
  deallocate(ffur,ffte, stat=merr)
 
  return
end subroutine mdffcor2
!
!--------------------------------------------------------------------
subroutine mdthndsk(de,pr,v1,v2,v3,x1,x2,x3,wok,pg,is1,ie1,js1,je1,ks1,ke1, &
                    tctd,rd,vd)
!--------------------------------------------------------------------
!  Geometrically Thin Disk Model
!
  use pram, only : imax, jmax, kmax, gam, c0, rbh, akm, metric
  implicit none 

  integer :: i, j, k, is1, ie1, js1, je1, ks1, ke1

  real(8) :: de(is1:ie1,js1:je1,ks1:ke1), pr(is1:ie1,js1:je1,ks1:ke1), &
             v1(is1:ie1,js1:je1,ks1:ke1), v2(is1:ie1,js1:je1,ks1:ke1), &
             v3(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: pg(is1:ie1,js1:je1,ks1:ke1)
  real(8) :: wok(3,is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: x1(imax), x2(jmax), x3(kmax)
!
  integer :: ikerr
  real(8) :: tctd, rd, vd, alitt, petty, aa, did, dd, xx, zz, rr, znor, &
             diskv, diskv2, diskd, dikv2, qqq, tmp1, xxx, gcm
  real(8) :: tmp1a, tmp1b
  real(8) :: ffjump
!
!--------------------------------------------------------------------
!  Parameter
!
  alitt=1.0d-2
  petty=1.0d-5
  aa=0.4d0
  did=0.25d0
  dd=0.25d0
  ikerr=0
  gcm=0.5*rbh
!
!--------------------------------------------------------------------
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1 
         
        xx=x1(i)*sin(x3(k))
        zz=x1(i)*cos(x3(k))
        rr=x1(i)
!
!     Disk Shape (transition) Setting
!
        if( xx.ne.0.d0 ) znor=abs(zz*rd/(xx*aa))
        if( xx.eq.0.d0 ) znor=1.d2
         
!        diskv=ffjump(xx,0.d0,1.d0,rd-did,did,2.0*did) &
!                *ffjump(znor,1.d0,0.d0,1.0+dd,dd,2.0*dd)
        tmp1a=ffjump(xx,1.d0,0.d0,rd-did,did,2.0*did)
        tmp1b=ffjump(znor,0.d0,1.d0,1.0+dd,dd,2.0*dd)
        diskv=ffjump(xx,1.d0,0.d0,rd-did,did,2.0*did) &
                *ffjump(znor,0.d0,1.d0,1.0+dd,dd,2.0*dd)

        dikv2=ffjump(xx,0.d0,vd,rd-did,did,2.0*did) &
                *ffjump(znor,1.d0,0.d0,1.0+dd,dd,2.0*dd)

!         diskd=ffjump(xx,0.d0,1.d0,rd+did,did,3.0*did)
!                *ffjump(znor,1.d0,0.d0,1.0-dd,dd,3.0*dd)
!         diskd=ffjump(xx,0.d0,1.d0,rd+did,did,2.0*did)
!                *ffjump(znor,1.d0,0.d0,1.0-dd,dd,3.0*dd)
        diskd=ffjump(xx,0.d0,1.d0,rd-did,did,2.0*did) &
                *ffjump(znor,1.d0,0.d0,1.0+dd,dd,2.0*dd)
     
!         zd=aa*xx/rd
!         diskv=ffjump(xx,0.0,1.0,rd-2.0*dd,dd,3.0*dd) &
!                *ffjump(abs(zz),1.0,0.0,zd,dd,3.0*dd)
!         diskd=ffjump(xx,0.0,1.0,rd+dd,dd,3.0*dd) &
!                *ffjump(abs(zz),1.0,0.0,zd-dd,dd,3.0*dd)
!
!  Density of Disk
!
        de(i,j,k)=de(i,j,k)+diskd*(tctd*de(i,j,k)-de(i,j,k))
!
!  Radial Velocity of Disk
!
        v1(i,j,k)=v1(i,j,k) &
                 *(1.0+diskv*(tctd*vd-1.0))/(1.0+diskv*(tctd-1.0))
!        v1(i,j,k)=v1(i,j,k)*diskv
!
!  Rotation Velocity of Disk
!        
        if( metric.le.203 .or. akm.eq.0.0 .or. ikerr.eq.0) then
!          write(*,*) "check OK1"          
          qqq=1.0+2.0*pg(i,j,k)/c0**2
!          tmp1=rr*qqq
          tmp1=xx*qqq
          v2(i,j,k)=dikv2*min(c0*(1.d0-alitt),sqrt(gcm/tmp1))
         
        elseif( ikerr.eq.1 ) then
          
          qqq=1.0+2.0*pg(i,j,k)/c0**2
          tmp1=rr*qqq
          v2(i,j,k)=dikv2*(min(c0*(1.d0-alitt),sqrt(gcm/tmp1) &
                        -c0*wok(2,i,j,k)))
!         v2(i,j,k)=v0*diskv*min(c0,sqrt(gcm/tmp1))
         
        elseif( ikerr.eq.2 ) then
          
          xxx=max( 1.5*(1.0+petty),rr/rbh )
          v2(i,j,k)=dikv2*(  c0/sqrt(2.0*(xxx-1.0))-c0*wok(2,i,j,k)  )
          v2(i,j,k)=min(vd*c0*(1.0-alitt),v2(i,j,k))
!
        else

        endif
!
        if(v2(i,j,k) .gt. 0.01d0) then
          v1(i,j,k)=0.d0
        endif

!          if(i .eq. 64 .and. j .eq. 4) then
!            write(*,*) x3(k), 'diskd=',diskd
!           write(*,*) 'de=',de(i,j,k), 'v1=',v1(i,j,k)
!          endif

      enddo
    enddo
  enddo
 
  return
end subroutine mdthndsk
!
!--------------------------------------------------------------------
subroutine mdtkdsk1(de,pr,v1,v2,v3,x1,x2,x3,hh,vp2,is1,ie1,js1,je1,ks1,ke1,&
                    vd,csd1)
!--------------------------------------------------------------------
! 
!     Thick Disk Option
!        Geometrically Thick Disk (Thick Torus)
!     ithdsk = 1 : constant angular momentum model 
!                  Abramowicz et al. 1978, A&A, 63, 221
!
  use pram, only : imax, jmax, kmax, gam, c0, rbh, akm, metric
  implicit none 

  integer :: i, j, k, is1, ie1, js1, je1, ks1, ke1

  real(8) :: de(is1:ie1,js1:je1,ks1:ke1), pr(is1:ie1,js1:je1,ks1:ke1), &
             v1(is1:ie1,js1:je1,ks1:ke1), v2(is1:ie1,js1:je1,ks1:ke1), &
             v3(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: hh(0:3,is1:ie1,js1:je1,ks1:ke1)
  real(8) :: vp2(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: x1(imax), x2(jmax), x3(kmax)

  real(8) :: wutin, clm, ckk, vd, v20, decut, redge, thedge, &
             thedge1, pr1, rh1, csd1, decor, prcor, v1cor, &
             rr, qt, rg, arg, del, sig, alp, gtt, grr, gpp, gtp, &
             gutt, gurr, gupp, gutp, &
             alpha1, omega3, beta3, wut, epthdsk, omg1, uut1, uuphi, alp1, &
             gam1, rco  
      
!--------------------------------------------------------------------
!  Parameter
!
!  wutin=-0.978d0
  wutin=-0.975d0
  clm=2.1d0
  ckk=1.0d-3
  v20=vd
  decut=15.d0
!
  redge=1.3d0
  thedge=0.783d0
  thedge1=2.3586d0
  pr1=0.d0
!
  csd1=0.d0
  gam1=1.d0/(gam-1.d0)
!
  rco=10.d0
!  wut=-1.d0
!
!--------------------------------------------------------------------
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1 

        decor=de(i,j,k)
        prcor=pr(i,j,k)
        v1cor=v1(i,j,k)
!   
        rr=x1(i)
        qt=x3(k)
! 
        if(metric .eq. 103) then
!          gtt=-(1.0-rbh/rr)
          gtt=-1.0*hh(0,i,j,k)**2
!          grr=1.0/(1.0-rbh/rr)
          grr=hh(1,i,j,k)**2
!          gpp=(rr*sin(qt))**2
          gpp=hh(2,i,j,k)**2
          gtp=0.d0
!          
          gutt=1.0/gtt
          gurr=(1.0-rbh/rr)
          gupp=1.0/(rr*sin(qt))**2
          gutp=0.d0
!        
!           alp1=sqrt(-gtt)
          alp1=hh(0,i,j,k)
          beta3=0.d0
!       
        elseif(metric .eq. 203) then
          rg=0.5*rbh
          arg=akm*rg
          del=rr**2.0-2.0*rg*rr+arg**2.0
          sig=rr**2.0+(arg*cos(qt))**2.0
          alp=(rr**2.0+arg**2.0)**2.0-del*(arg*sin(qt))**2.0
!
          gtt=-(1.0-(2.0*rg*rr/sig))
          gtp=-(2.0*arg*rg*rr*sin(qt)**2.0)/sig

!          grr=sig/del
          grr=hh(1,i,j,k)**2
!          gpp=(alp*sin(qt)**2.0)/sig
          gpp=hh(2,i,j,k)**2
        
          gutt=-alp/(del*sig)
          gurr=del/sig
          gutp=-(2.0*arg*rg*rr)/(sig*del)
          gupp=(sig-arg**2*sin(qt)**2)/(del*sig*sin(qt)**2)
          
          alp1=sqrt(sig*del/alp)
          omega3=2.0*arg*rg*rr/alp
          beta3=(sqrt(alp/sig)*sin(qt)*omega3)/alp1
        
        endif
!
!      constant angular momentum model (clm=const)  
!      Abramowicz et al. (1978)
!
        wut=-1.0*sqrt(abs((-gtp**2+gtt*gpp) &
              /(gpp+2.0*clm*gtp+gtt*clm**2)))
!        wut=-1.0/sqrt(abs(gutt-2.0*gutp*clm+gupp*clm**2.0))

!        if(j .eq. 4 .and. k .eq. 128) then
!          write(*,*) 'wut,rr', wut, rr
!        endif 
!        if(wut .gt. wutin) then
        if(wut .gt. wutin .and. rr .ge. redge & 
          .and. qt .ge. thedge .and. qt .le. thedge1) then
          epthdsk=((wutin/wut)-1.0)/gam
!          
          de(i,j,k)=(epthdsk*(gam-1.0)/ckk)**gam1
!          pr(i,j,k)=ckk*(epthdsk*(gam-1.0)/ckk)**(gam/(gam-1.0))
          pr(i,j,k)=ckk*de(i,j,k)**gam

!        if(j .eq. 4 .and. k .eq. 128) then
!          write(*,*) 'wut,rr', wut,rr
!          write(*,*) 'de,pr,rr', de(i,j,k),pr(i,j,k), rr
!        endif 

        else
!          de(i,j,k)=0.1d0*exp(-0.5d0*rr/rco)
          de(i,j,k)=1.0d-2
          pr(i,j,k)=ckk*de(i,j,k)**gam

        endif
!
!     Vector potential for poloidal loop field
!
        if(de(i,j,k) .ge. decut .and. rr .ge. redge) then
          vp2(i,j,k)=(de(i,j,k)-decut)
        else
          vp2(i,j,k)=0.d0
        endif
! 
      
        if(wut .gt. wutin .and. rr .ge. redge &
            .and. qt .ge. thedge .and. qt .le. thedge1) then
          v1(i,j,k)=0.d0
          v3(i,j,k)=0.d0
          omg1=-1.0*(clm*gtt+gtp)/(clm*gtp+gpp)
          uut1=-1.0/(wut*(1.0-clm*omg1))
          uuphi=omg1*uut1
          v2(i,j,k)=0.3*vd*(hh(2,i,j,k)*(uuphi/(alp1*uut1))-beta3)
!          v2(i,j,k)=0.d0
        else
!          v1(i,j,k) = v1cor
          v1(i,j,k) = 0.d0
          v2(i,j,k) = 0.d0
          v3(i,j,k) = 0.d0
        endif
!
!   Calculation for perturbation
!
!        if(x2(j) .eq. 0) then
!          if(pr(i,j,k) .gt. pr1) then
!            pr1=pr(i,j,k)
!            rh1=de(i,j,k)+(gam/(gam-1.0))*pr(i,j,k)
!            csd1=sqrt(gam*pr(i,j,k)/rh1)
!          endif
!        endif
!
      enddo
    enddo
  enddo 
!
  return
end subroutine mdtkdsk1
!
!--------------------------------------------------------------------
subroutine mdtkdsk2(de,pr,v1,v2,v3,x1,x2,x3,hh,vp2,is1,ie1,js1,je1,ks1,ke1, &
                    vd,csd1)
!--------------------------------------------------------------------
! 
!     Thick Disk Option
!        Geometrically Thick Disk (Thick Torus)
!     ithdsk = 2 : power-law rotation model (nearly Keplerian)
!                  De Villiers et al. 2003, ApJ, 599, 1238
!
  use pram, only : imax, jmax, kmax, gam, c0, rbh, akm, metric, pi, iter
  implicit none 

  integer :: i, j, k, is1, ie1, js1, je1, ks1, ke1, nnn

  real(8) :: de(is1:ie1,js1:je1,ks1:ke1), pr(is1:ie1,js1:je1,ks1:ke1), &
             v1(is1:ie1,js1:je1,ks1:ke1), v2(is1:ie1,js1:je1,ks1:ke1), &
             v3(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: hh(0:3,is1:ie1,js1:je1,ks1:ke1)
  real(8) :: vp2(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: x1(imax), x2(jmax), x3(kmax)
  
  real(8) :: wutin, clm, ckk, vd, v20, decut, qtin, redge, thedge, &
             thedge1, pr1, rh1, csd1, decor, prcor, v1cor, &
             rr, qt, rg, arg, del, sig, alp, gtt, grr, gpp, gtp, &
             gutt, gurr, gupp, gutp, alp1, omega3, beta3, &
               guttin, gurrin, guppin, gutpin, plmdinsq, delin, sigin, alpin, &
             plmdin, etaa1, ckk1, calp, calp2, flin, tmp1, plmdsq, plmd, &
             clm1, fl1, cldel, cldelend, clf1, cldf1, cqq1, tmp1a, tmp1b, &
             xx_old, wut, tmp2, epthdsk, omg1, uut1, uuphi, cqq, rrin
!
  !--------------------------------------------------------------------
!  Parameter
!
  wutin=-0.9696d0
  clm=2.22d0
  ckk=1.0d-4
  v20=vd
  decut=15.d0

  qtin=pi/2.d0
      
  redge=4.d0
  thedge=0.783d0
  thedge1=2.3586d0
!
!--------------------------------------------------------------------
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1 

        decor=de(i,j,k)
        prcor=pr(i,j,k)
        v1cor=v1(i,j,k)
!   
        rr=x1(i)
        qt=x3(k)
       
        if(metric .eq. 103) then
!          gtt=-(1.0-rbh/rr)
          gtt=-1.0*hh(0,i,j,k)**2
!          grr=1.0/(1.0-rbh/rr)
          grr=hh(1,i,j,k)**2
!          gpp=(rr*sin(qt))**2
          gpp=hh(2,i,j,k)**2
          gtp=0.d0
          
          gutt=1.0/gtt
          gurr=(1.0-rbh/rr)
          gupp=1.0/(rr*sin(qt))**2
          gutp=0.d0

!          alp1=sqrt(-gtt)
          alp1=hh(0,i,j,k)
          beta3=0.d0
       
        elseif(metric .eq. 203) then
          rg=0.5*rbh
          arg=akm*rg
          del=rr**2.0-2.0*rg*rr+arg**2.0
          sig=rr**2.0+(arg*cos(qt))**2.0
          alp=(rr**2.0+arg**2.0)**2.0-del*(arg*sin(qt))**2.0
!
          gtt=-(1.0-(2.0*rg*rr/sig))
          gtp=-(2.0*arg*rg*rr*sin(qt)**2.0)/sig
!          grr=sig/del
          grr=hh(1,i,j,k)**2
!          gpp=(alp*sin(qt)**2.0)/sig
          gpp=hh(2,i,j,k)**2
         
          gutt=-alp/(del*sig)
          gurr=del/sig
          gutp=-(2.0*arg*rg*rr)/(sig*del)
          gupp=(sig-arg**2*sin(qt)**2)/(del*sig*sin(qt)**2)
        
          alp1=sqrt(sig*del/alp)
          omega3=2.0*arg*rg*rr/alp
          beta3=(sqrt(alp/sig)*sin(qt)*omega3)/alp1
        endif
!
!      thick torus with power-law rotation
!      De Villiers et al. (2004)
!      addtional initial parameter, rrin, cqq
!
        if(metric .eq. 103) then
        
          guttin=-1.0/(1.0-rbh/rrin)
          gurrin=(1.0-rbh/rrin)
          guppin=1.0/(rrin*sin(qtin))**2
          gutpin=0.d0
          
          plmdinsq=-guttin/guppin
         
        elseif(metric .eq. 203) then
        
          delin=rrin**2.0-2.0*rg*rrin+arg**2.0
          sigin=rrin**2.0+(arg*cos(qtin))**2.0
          alpin=(rrin**2.0+arg**2.0)**2.0-delin*(arg*sin(qtin))**2.0
!
          guttin=-alpin/(delin*sigin)
          gurrin=delin/sigin
          gutpin=-(2.0*arg*rg*rrin)/(sigin*delin)
          guppin=(sigin-arg**2*sin(qtin)**2)/(delin*sigin*sin(qtin)**2)
         
          plmdinsq=clm*((guttin-clm*gutpin)/(gutpin-clm*guppin))
         
        endif
         
        plmdin=sqrt(plmdinsq)
        etaa1=clm/plmdin**(2.0-cqq)
        ckk1=etaa1**(-2.0/(cqq-2.0))
        calp=cqq/(cqq-2.0)
        calp2=calp+1.0
        flin=abs(1.0-ckk1*clm**calp2)**(1.0/calp2)
        
        tmp1=wutin*flin
        
!         if(i .eq. 20 .and. j .eq. 4 .and. k .eq. 120) then
!          write(*,*) 'plmdin, flin=', plmdin, flin
!          write(*,*) 'etaa1, ckk1=', etaa1, ckk1
!          write(*,*) 'tmp1=', tmp1
!         endif 
        
        if(metric .eq. 103) then
          plmdsq=-gutt/gupp
          plmd=sqrt(plmdsq)
          clm1=etaa1*plmd**(2.0-cqq)
          fl1=abs(1.0-ckk1*clm1**calp2)**(1.0/calp2)
        
!         if(x1(i) .ge. 3.0 .and. x1(i) .le. 10.0  
!     &      .and. j .eq. 4 .and. k .eq. 120) then
!          write(*,*) 'x1(i)=', x1(i)
!          write(*,*) 'plmd, fl1=', plmd, fl1
!         endif 
        
        elseif(metric .eq. 203 .or. metric .eq. 303) then
          cldel=1.d0
          cldelend=0.001d0
          clf1=1.d0
          cldf1=1.d0
          nnn=0
         
          plmdsq=clm*(gutt-clm*gutp)/(gutp-clm*gupp)
          plmd=sqrt(plmdsq)
          clm1=etaa1*plmd**(2.0-cqq)
         
          cqq1=(2.0-cqq)/2.0
         
! ---
 95       continue
          if(abs(cldel) .lt. cldelend) then
            tmp1a=clm1*(gutp-clm1*gupp)**cqq1
            tmp1b=-etaa1*(clm1*gutt-gutp*clm1**2)**cqq1
            clf1=tmp1a+tmp1b
              
            cldf1=((gutp-clm1*gupp)**cqq1)* &
                  (-gupp*clm1*cqq1*(gutp-gupp*clm1)**(-cqq/2.0)) &
                  -etaa1*cqq1*(gutt-2.0*clm1*gutp) &
                  *(clm1*gutt-gutp*clm1**2)**(-cqq/2.0)
          
            if(clf1 .eq. 0.0 .or. cldf1 .eq. 0.0) then
              write(6,*) '>> Bad numerical divergence in thick disk' &
                         ,'at i,j,k:', i, j, k
              write(6,*) '>> f1, df1 =', clf1, cldf1,'at i,j,k:', i, j, k
            else
              cldel=clf1/cldf1
            endif
          
            xx_old=clm1
            clm1=clm1-cldel
          
            if(nnn .gt. iter) then
              write(6,*) 'nnn > iter'
              stop
            endif
          
            nnn=nnn+1
            goto 95
          endif
! ----
          fl1=abs(1.0-ckk1*clm1**calp2)**(1.0/calp2)
          
!          if(x1(i) .ge. 3.0 .and. x1(i) .le. 10.0 &  
!            .and. j .eq. 4 .and. k .eq. 120) then
!           write(*,*) 'x1(i)=', x1(i)
!           write(*,*) 'clm1, fl1=', clm1, fl1
!          endif 

        endif
        
        wut=-1.0*sqrt(abs((gtp**2+gtt*gpp) &
            /(gpp+2.0*clm1*gtp+gtt*clm1**2)))
        tmp2=wut*fl1
        
!         if(x1(i) .ge. 3.0 .and. x1(i) .le. 10.0  &
!           .and. j .eq. 4 .and. k .eq. 120) then
!          write(*,*) 'wut, tmp2=', wut, tmp2
!         endif 
        
        if(tmp2 .gt. tmp1 .and. rr .ge. redge  &
           .and. qt .ge. thedge .and. qt .le. thedge1) then
          epthdsk=((tmp1/tmp2)-1.0)/gam
          de(i,j,k)=(epthdsk*(gam-1.0)/ckk)**(1.0/(gam-1.0))
!          pr(i,j,k)=ckk*(epthdsk*(gam-1.0)/ckk)**(gam/(gam-1.0))
          pr(i,j,k)=ckk*de(i,j,k)**gam
       
          if(de(i,j,k) .le. decor) then
            de(i,j,k)=decor
          endif
          if(pr(i,j,k) .le. prcor) then
            pr(i,j,k)=prcor
          endif
        else
          de(i,j,k)=decor
          pr(i,j,k)=prcor
        endif
!
!       Vector potential for poloidal loop field
!
        if(de(i,j,k) .ge. decut .and. rr .ge. redge) then
          vp2(i,j,k)=(de(i,j,k)-decut)
        else
          vp2(i,j,k)=0.d0
        endif
!
        if(tmp2 .gt. tmp1 .and. rr .ge. redge & 
           .and. qt .ge. thedge .and. qt .le. thedge1) then
          v1(i,j,k)=0.d0
          v3(i,j,k)=0.d0
          omg1=-1.0*(clm1*gtt+gtp)/(clm1*gtp+gpp)
          uut1=-1.0/(wut*(1.0-clm1*omg1))
          uuphi=omg1*uut1
          v2(i,j,k)=v20*(hh(2,i,j,k)*(uuphi/(alp1*uut1))-beta3)
        else
          v1(i,j,k) = v1cor
          v2(i,j,k) = 0.d0
          v3(i,j,k) = 0.d0
        endif
!
!   Calculation for perturbation
!
        if(x2(j) .eq. 0.d0) then
          if(pr(i,j,k) .gt. pr1) then
            pr1=pr(i,j,k)
            rh1=de(i,j,k)+(gam/(gam-1.0))*pr(i,j,k)
            csd1=sqrt(gam*pr(i,j,k)/rh1)
          endif
        endif
         
      enddo
    enddo
  enddo 

  return
end subroutine mdtkdsk2
!
!--------------------------------------------------------------------
subroutine mdmonomg1(b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1,b0)
!--------------------------------------------------------------------
!     Monopole Magnetic Field
!
  use pram, only : imax, jmax, kmax, gam, c0
  implicit none

  integer :: i, j, k, is1, ie1, js1, je1, ks1, ke1
!
  real(8) :: b1(is1:ie1,js1:je1,ks1:ke1), b2(is1:ie1,js1:je1,ks1:ke1), &
             b3(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: x1(imax), x2(jmax), x3(kmax)
!
  real(8) :: b0, a1
!
!--------------------------------------------------------------------
!  Parameter
!
      a1=-2.d0
!
!-------------------------------------------------------------------
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1 

        b1(i,j,k)=b0*x1(i)**a1
        b2(i,j,k)=0.0
        b3(i,j,k)=0.0

      enddo
    enddo
  enddo
 
  return
end subroutine mdmonomg1
!
!--------------------------------------------------------------------
subroutine mdmonomg2(b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1,b0)
!--------------------------------------------------------------------
!     Monopole Magnetic Field (for Kerr Black Hole)
!
  use pram, only : imax, jmax, kmax, gam, c0, akm, rbh
  implicit none

  integer :: i, j, k, is1, ie1, js1, je1, ks1, ke1
!
  real(8) :: b1(is1:ie1,js1:je1,ks1:ke1), b2(is1:ie1,js1:je1,ks1:ke1), &
             b3(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: x1(imax), x2(jmax), x3(kmax)
!
  real(8) :: b0, rr, qt, rg, arg, del, sig, alp
!
!--------------------------------------------------------------------
!
  rg=0.5*rbh
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1 

        rr=x1(i)
        qt=x3(k)
        arg=akm*rg

        del=rr**2-2.0*rg*rr+arg**2
        sig=rr**2+(arg*cos(qt))**2
        alp=(rr**2+arg**2)**2-del*(arg*sin(qt))**2

        b1(i,j,k)=b0*(rr*rr+arg*arg)*(rr*rr-(arg*cos(qt))**2) &
                  /(sig*sig*sqrt(alp))
        b2(i,j,k)=0.d0
        b3(i,j,k)=-b0*2.0*rr*arg*arg*sin(qt)*cos(qt) &
                  *sqrt(del/alp)/(sig*sig)

      enddo
    enddo
  enddo
 
  return
end subroutine mdmonomg2
!
!--------------------------------------------------------------------
subroutine mddipmg(b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1,b0)
!--------------------------------------------------------------------
!     Dipole Magnetic Field
!
  use pram, only : imax, jmax, kmax, gam, c0
  implicit none

  integer :: i, j, k, is1, ie1, js1, je1, ks1, ke1
!
  real(8) :: b1(is1:ie1,js1:je1,ks1:ke1), b2(is1:ie1,js1:je1,ks1:ke1), &
             b3(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: x1(imax), x2(jmax), x3(kmax)
!
  real(8) :: b0
!
!--------------------------------------------------------------------
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1 

        b1(i,j,k)=2.0*b0*cos(x3(k))/x1(i)**3.0
        b2(i,j,k)=0.d0
        b3(i,j,k)=b0*sin(x3(k))/x1(i)**3.0

      enddo
    enddo
  enddo  
!
  return
end subroutine mddipmg
!
!--------------------------------------------------------------------
subroutine mdpotmg(b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1,b0)
!--------------------------------------------------------------------
!     Potential Magnetic Field
!
!    From Ouyed & Pudritz 1997, ApJ, 482, 712
!    zzd is dimensionless disk thickness
!    zzd < 0.1 for a thin Keplerian disk
!
  use pram, only : imax, jmax, kmax, gam, c0
  implicit none

  integer :: i, j, k, is1, ie1, js1, je1, ks1, ke1
!
  real(8) :: b1(is1:ie1,js1:je1,ks1:ke1), b2(is1:ie1,js1:je1,ks1:ke1), &
             b3(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: x1(imax), x2(jmax), x3(kmax)
!
  real(8) :: b0, zzd, rr, zz, br, bz
!
!-------------------------------------------------------------------
!  Parameter
!
  zzd=0.1d0
!
!--------------------------------------------------------------------
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1 
        
        rr=x1(i)*sin(x3(k))
        zz=x1(i)*cos(x3(k))

        br=b0*(1.0-((zzd+zz)/sqrt(rr**2.0+(zzd+zz)**2.0)))/rr
        bz=b0/sqrt(rr**2.0+(zzd+zz)**2.0)
!
        b1(i,j,k)=br*sin(x3(k))+bz*cos(x3(k))
        b2(i,j,k)=0.d0
        b3(i,j,k)=br*cos(x3(k))-bz*sin(x3(k))


      enddo
    enddo
  enddo
!
  return
end subroutine mdpotmg
!
!--------------------------------------------------------------------
subroutine mdsmonmg(b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1,b0)
!--------------------------------------------------------------------
!    Split Monopole Field
!
!    From Sakurai 1987, PASJ, 39, 821
!    zzd1 is depth from equatorial plane, zzd1 > 0.0
!
  use pram, only : imax, jmax, kmax, gam, c0
  implicit none

  integer :: i, j, k, is1, ie1, js1, je1, ks1, ke1
!
  real(8) :: b1(is1:ie1,js1:je1,ks1:ke1), b2(is1:ie1,js1:je1,ks1:ke1), &
             b3(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: x1(imax), x2(jmax), x3(kmax)
!
  real(8) :: b0, zzd, rr, zz, br, bz, zzd1
!
!-------------------------------------------------------------------
!  Parameter
!
  zzd1=2.0d0
!
!--------------------------------------------------------------------
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1 

        rr=x1(i)*sin(x3(k))
        zz=x1(i)*cos(x3(k))

        br=b0*rr*(rr**2.0+(zz+zzd1)**2.0)**(-3.0/2.0)
        bz=b0*(zz+zzd1)*(rr**2.0+(zz+zzd1)**2.0)**(-3.0/2.0)

        b1(i,j,k)=br*sin(x3(k))+bz*cos(x3(k))
        b2(i,j,k)=0.d0
        b3(i,j,k)=br*cos(x3(k))-bz*sin(x3(k))

      enddo
    enddo
  enddo
!
  return
end subroutine mdsmonmg
!
!--------------------------------------------------------------------
subroutine mdunimg1(b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1,b0,pg)
!--------------------------------------------------------------------
!
!      Uniform Vertical Magnetic Field (Wald Solution)
!
  use pram, only : imax, jmax, kmax, gam, c0
  implicit none

  integer :: i, j, k, is1, ie1, js1, je1, ks1, ke1
!
  real(8) :: b1(is1:ie1,js1:je1,ks1:ke1), b2(is1:ie1,js1:je1,ks1:ke1), &
             b3(is1:ie1,js1:je1,ks1:ke1)
  real(8) :: pg(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: x1(imax), x2(jmax), x3(kmax)
!
  real(8) :: b0
!
!--------------------------------------------------------------------
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1 

        b1(i,j,k)=b0*cos(x3(k))
        b2(i,j,k)=0.d0
        b3(i,j,k)=-b0*sin(x3(k))*sqrt(1.0+2.0*pg(i,j,k)/c0**2)

      enddo
    enddo
  enddo

  return
end subroutine mdunimg1
!
!--------------------------------------------------------------------
subroutine mdunimg2(b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1,b0)
!--------------------------------------------------------------------
  use pram, only : imax, jmax, kmax, gam, c0, rbh, akm
  implicit none

  integer :: i, j, k, is1, ie1, js1, je1, ks1, ke1
!
  real(8) :: b1(is1:ie1,js1:je1,ks1:ke1), b2(is1:ie1,js1:je1,ks1:ke1), &
             b3(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: x1(imax), x2(jmax), x3(kmax)
!
  real(8) :: b0, rr, qt, del, sig, alp, rg, arg
!
!-------------------------------------------------------------------
!
  rg=0.5*rbh
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1      
        
        rr=x1(i)
        qt=x3(k)
        arg=akm*rg

        del=rr**2-2.0*rg*rr+arg**2
        sig=rr**2+(arg*cos(qt))**2
        alp=(rr**2+arg**2)**2-del*(arg*sin(qt))**2

!        b1(i,j,k)=b1(i,j,k)+
        b1(i,j,k)=b0/sqrt(alp)*cos(qt)*( del &
                 +2.0*rg*rr*(rr**2-arg**2)*(rr**2+arg**2)/sig**2 )
        b2(i,j,k)=0.d0
         
!        b3(i,j,k)=b3(i,j,k)+
        b3(i,j,k)=-b0*sqrt(del/alp)*sin(qt) &
                 *(rr-rg+rg/sig**2* &
                 ((rr**2+arg**2)*sig+2.0*(arg*cos(qt))**2*(rr**2-arg**2)))

      enddo
    enddo
  enddo

  return
end subroutine mdunimg2
!
!--------------------------------------------------------------------
subroutine mdvpmg1(b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1,b0,hh,vp2)
!--------------------------------------------------------------------
!      Magnetic Field Calculated from Vector Potential
!      (poloidal loop field in the thick torus: thick disk model)
!      set vp2
!
  use pram, only : imax, jmax, kmax, gam, c0
  implicit none

  integer :: i, j, k, is1, ie1, js1, je1, ks1, ke1
!
  real(8) :: b1(is1:ie1,js1:je1,ks1:ke1), b2(is1:ie1,js1:je1,ks1:ke1), &
             b3(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: hh(0:3,is1:ie1,js1:je1,ks1:ke1)
  real(8) :: vp2(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: x1(imax), x2(jmax), x3(kmax)
!
  real(8) :: b0, rr, qt, gtt, grr, gpp, gthth, alp1, det
!
!-------------------------------------------------------------------
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1   

        rr=x1(i)
        qt=x3(k)

!        gtt=-(1.0-rbh/rr)
        gtt=-1.0*hh(0,i,j,k)**2
!        grr=1.0/(1.0-rbh/rr)
        grr=hh(1,i,j,k)**2
!        gpp=(rr*sin(qt))**2
        gpp=hh(2,i,j,k)**2
        gthth=hh(3,i,j,k)**2
        alp1=sqrt(-gtt)
        det=alp1*sqrt(grr*gpp*gthth)

        if(k .eq. 0) then
         b1(i,j,k)=(b0/det)*(vp2(i,j,k+1)-vp2(i,j,k))/(x3(k+1)-x3(k))
        elseif(k .eq. kmax) then
          b1(i,j,k)=(b0/det)*(vp2(i,j,k)-vp2(i,j,k-1))/(x3(k)-x3(k-1))
        else         
          b1(i,j,k)=(b0/det)*(vp2(i,j,k+1)-vp2(i,j,k-1))/(x3(k+1)-x3(k-1))
        endif
!
        if(i .eq. 0) then
          b2(i,j,k)=-(b0/det)*(vp2(i+1,j,k)-vp2(i,j,k))/(x1(i+1)-x1(i))
        elseif(i .eq. imax) then
          b2(i,j,k)=-(b0/det)*(vp2(i,j,k)-vp2(i-1,j,k))/(x1(i)-x1(i-1))
        else         
          b2(i,j,k)=-(b0/det)*(vp2(i+1,j,k)-vp2(i-1,j,k))/(x1(i+1)-x1(i-1))
        endif

        b3(i,j,k)=0.d0 
!
      enddo
    enddo
  enddo

  return
end subroutine mdvpmg1
!
!--------------------------------------------------------------------
subroutine mdvpmg2(b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1,b0,vp2)
!--------------------------------------------------------------------
!      Magnetic Field Calculated from Vector Potential
!      (poloidal loop field in the thick torus: thick disk model)
!      set vp2
!
  use pram, only : imax, jmax, kmax, gam, c0, rbh, akm
  implicit none

  integer :: i, j, k, is1, ie1, js1, je1, ks1, ke1
!
  real(8) :: b1(is1:ie1,js1:je1,ks1:ke1), b2(is1:ie1,js1:je1,ks1:ke1), &
             b3(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: vp2(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: x1(imax), x2(jmax), x3(kmax)
!
  real(8) :: b0, rr, qt, rg, arg, del, sig, alp, gtt, gtp, grr, gpp, gthth, &
             alp1, det
!
!-------------------------------------------------------------------
!
  rg=0.5*rbh
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1  

        rr=x1(i)
        qt=x3(k)
        arg=akm*rg

        del=rr**2.0-2.0*rg*rr+arg**2.0
        sig=rr**2.0+(arg*cos(qt))**2.0
        alp=(rr**2.0+arg**2.0)**2.0-del*(arg*sin(qt))**2.0
!
        gtt=-(1.0-(2.0*rg*rr/sig))
        gtp=-(2.0*arg*rg*rr*sin(qt)**2.0)/sig

        grr=sig/del
        gpp=(alp*sin(qt)**2.0)/sig
        gthth=sig
        alp1=sqrt(sig*del/alp)
        det=alp1*sqrt(grr*gpp*gthth)

        if(k .eq. 0) then
          b1(i,j,k)=(b0/det)*(vp2(i,j,k+1)-vp2(i,j,k))/(x3(k+1)-x3(k))
        elseif(k .eq. kmax) then
          b1(i,j,k)=(b0/det)*(vp2(i,j,k)-vp2(i,j,k-1))/(x3(k)-x3(k-1))
        else         
          b1(i,j,k)=(b0/det)*(vp2(i,j,k+1)-vp2(i,j,k-1))/(x3(k+1)-x3(k-1))
        endif
        
        if(i .eq. 0) then
          b2(i,j,k)=-(b0/det)*(vp2(i+1,j,k)-vp2(i,j,k))/(x1(i+1)-x1(i))
        elseif(i .eq. imax) then
          b2(i,j,k)=-(b0/det)*(vp2(i,j,k)-vp2(i-1,j,k))/(x1(i)-x1(i-1))
        else         
          b2(i,j,k)=-(b0/det)*(vp2(i+1,j,k)-vp2(i-1,j,k))/(x1(i+1)-x1(i-1))
        endif
        b3(i,j,k)=0.d0 

      enddo
    enddo
  enddo

  return
end subroutine mdvpmg2
!
!--------------------------------------------------------------------
      subroutine mdloopmg(b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1,b0)
!--------------------------------------------------------------------
!
!      Reversed Current Loop Magnetic Field               2005.03.08
!
  use pram, only : imax, jmax, kmax, gam, c0, rbh, akm, xmin
  implicit none

  integer :: i, j, k, is1, ie1, js1, je1, ks1, ke1
!
  real(8) :: b1(is1:ie1,js1:je1,ks1:ke1), b2(is1:ie1,js1:je1,ks1:ke1), &
             b3(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: x1(imax), x2(jmax), x3(kmax)

  integer :: itymag, nma, nua
  real(8) :: b0, aj0, rr0, a0, dd0, rhm, rbm, &
             alamb, rth, amu0, vrsm, rr, qt, arg, del, sig, alp, &
             rn, hm, bm, xm, rcm, rthm, rm, rdm, pm, cm, qm, amu, &
             um, udhm, aaa, ajf, rg
  real(8) :: ffjump 
!
!-------------------------------------------------------------------
!
!  Parameter  
!
  rg=0.5*rbh
  itymag=2
  aj0=1.5d-1
  rr0=6.d0
  a0=1.d0
  dd0=0.1d0
  rhm=4.d0
  rbm=8.d0
  alamb=1.d0
  rth=1.d0
  amu0=0.5d0
  nua=5
!
  vrsm=1.0d-5
!
!-------------------------------------------------------------------
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1  
        
        rr=x1(i)
        qt=x3(k)
        arg=akm*rg
        del=rr**2-2.0*rg*rr+arg**2
        sig=rr**2+(arg*cos(qt))**2
        alp=(rr**2+arg**2)**2-del*(arg*sin(qt))**2
!
        if( itymag.eq.1) then
          b1(i,j,k)=b0/sqrt(alp)*cos(qt)*( del &
                   +2.0*rg*rr*(rr**2-arg**2)*(rr**2+arg**2)/sig**2 )
          b2(i,j,k)=0.d0
          b3(i,j,k)=-b0*sqrt(del/alp)*sin(qt)*(  rr-rg+rg/sig**2* &
            ( (rr**2+arg**2)*sig+2.0*(arg*cos(qt))**2*(rr**2-arg**2) )  )
        else
          b1(i,j,k)=0.d0
          b2(i,j,k)=0.d0
          b3(i,j,k)=0.d0
        endif
!
        if( rhm.lt.rr0 .and. rbm.gt.rr0 ) then
!tae       rn=rr*sqrt(sin(qt)**2+(cos(qt)/alamb)**2)
!tae       cosqn=cos(qt)*rr/(alamb*rn)
!tae       sinqn=sin(qt)*rr/rn
          rn=rr
!
          hm=rr0-rhm
          bm=rbm-rr0
          xm=rn-rr0
!
          if( bm*(1.0-vrsm).lt.hm .and. hm.lt.bm*(1.0+vrsm) ) then
            rcm=rhm+a0
         
          elseif( bm.gt.hm ) then
!x           rthm=a0*( rr0/bm*(bm/hm-1.0)/log(bm/hm)-1.0 )
!x           rthm=a0*( alamb*rr0/bm*(bm/hm-1.0)/log(bm/hm)-1.0 )
            rthm=rth
          
            if( rthm.lt.0.d0 ) then
              write(6,*) 'ERROR: rthm < 0; rthm =',rthm
              write(6,*) 'STOP at main'
              stop
            endif
          
            rcm=rthm+a0
          
          else
            write(6,*) 'Initial Setting of Magnetic Field Error!'
            write(6,*) 'for Sora-mame type configuration: bm > hm'
            write(6,*) ' bm, hm: ',bm,hm
            write(6,*) 'Stop at main'
            stop
          endif
!
          if( bm*(1.0-vrsm).lt.hm .and. hm.lt.bm*(1.0+vrsm) ) then
            rm=rn
            rdm=1.d0
         
          elseif( bm.gt.hm ) then

!d           if( rhm.le.rn .and. rn.le.rbm ) then
            if( rn.gt.(rbm*rhm-rr0*rr0)/(rbm+rhm-2.0*rr0)+vrsm ) then
              pm=hm*bm/(bm-hm)
              cm=a0/log(bm/hm)
              qm=rthm-a0*log(hm*hm/(bm-hm))/log(bm/hm) 
              rm=cm*log(xm+pm)+qm
              rdm=cm/(xm+pm)
            elseif( rhm.gt.rn ) then
!d              rm=rthm
              rm=xmin+vrsm
              rdm=0.d0
            else
!d             rm=rthm+2.0*a0
              rm=xmin+vrsm
              rdm=0.d0
            endif
         
          else
            write(6,*) 'Initial Setting of Magnetic Field Error!'
            write(6,*) 'for Sora-mame type configuration: bm > hm'
            write(6,*) ' bm, hm: ',bm,hm
            write(6,*) 'Stop at main'
            stop
          endif

!tae        aaa=sqrt(rcm**2+rm**2-2.0*rm*rcm*sinqn)
          amu=1.0+(amu0-1.0)*cos(qt)**(2*nma)
          um=(cos(qt)/alamb)**2+(sin(qt)/amu)**2
          udhm=sin(qt)*cos(qt)*(1.0/amu**2-1.0/alamb**2 &
              -(1.0-amu0)*2.0*nma*sin(qt)**2*cos(qt)**(2*nma-2)/amu**3)
         
          aaa=sqrt(um*rm**2+rcm**2-2.0*rm*rcm*sin(qt)/amu)
          ajf=aj0*ffjump(aaa,1.d0,0.d0,a0-3.2*dd0,dd0,3.0*dd0)

          b1(i,j,k)=b1(i,j,k)+0.5*rcm*ajf/(sqrt(alp)*sin(qt)) &
                   *rm*( rcm*cos(qt)/amu-rm*udhm &
                   -rcm*sin(qt)/amu**2 &
                   *2.0*nma*(1.0-amu0)*cos(qt)**(2*nma-1)*sin(qt))
          b2(i,j,k)=b2(i,j,k)
          b3(i,j,k)=b3(i,j,k)+0.5*rcm*ajf*sqrt(del/alp)/sin(qt) &
                   *rdm*(rm*um-rcm*sin(qt)/amu)

!tae       b1=b1+0.5*rcm*ajf/(sqrt(alp)*sin(qt))
!tae &        *(rm*rcm*cosqn*(rr/rn)**2/alamb
!tae &        -(rm-rcm*sinqn)*rdm*(alamb**2-1.0)/alamb*rn*sinqn*cosqn)
!tae       b2=b2
!tae       b3=b3+0.5*rcm*rdm*(rm-rcm*sinqn)*ajf*sqrt(del/alp)/sin(qt)
!tae &        *rn/rr

        else

          aaa=sqrt(rr0**2+rr**2-2.0*rr*rr0*sin(qt))
!          ajf=aj0*ffjump(aaa,1.d0,0.d0,a0,dd0,3.0*dd0)
          ajf=aj0*ffjump(aaa,1.d0,0.d0,a0-3.0*dd0,dd0,3.0*dd0)

          b1(i,j,k)=b1(i,j,k)+0.5*rr*rr0**2*cos(qt) &
                   *ajf/(sqrt(alp)*sin(qt))
          b2(i,j,k)=b2(i,j,k)
          b3(i,j,k)=b3(i,j,k)+0.5*rr0*(rr-rr0*sin(qt)) &
                  *ajf*sqrt(del/alp)/sin(qt)
        endif
       
      enddo
    enddo
  enddo
!
  return
end subroutine mdloopmg
!
!--------------------------------------------------------------------
subroutine mdadmnmg(b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1,qmag)
!--------------------------------------------------------------------
!
!      Reversed Current Loop Magnetic Field               2005.03.08
!
  use pram, only : imax, jmax, kmax, gam, c0, rbh, akm
  implicit none

  integer :: i, j, k, is1, ie1, js1, je1, ks1, ke1
!
  real(8) :: b1(is1:ie1,js1:je1,ks1:ke1), b2(is1:ie1,js1:je1,ks1:ke1), &
             b3(is1:ie1,js1:je1,ks1:ke1)
!
  real(8) :: x1(imax), x2(jmax), x3(kmax)
!
  real(8) :: qmag, rr, qt, arg, del, sig, alp, rg
!
!-------------------------------------------------------------------
!
  rg=0.5*rbh
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1  

        rr=x1(i)
        qt=x3(k)
        arg=akm*rg

        del=rr**2.0-2.0*rg*rr+arg**2.0
        sig=rr**2.0+(arg*cos(qt))**2.0
        alp=(rr**2.0+arg**2.0)**2.0-del*(arg*sin(qt))**2.0
!
        b1(i,j,k)=b1(i,j,k)+qmag*(rr*rr+arg*arg) &
                 *(rr*rr-(arg*cos(qt))**2)/(sig*sig*sqrt(alp))
        b2(i,j,k)=b2(i,j,k)
        b3(i,j,k)=b3(i,j,k)-qmag*2.0*rr*arg*arg*sin(qt)*cos(qt) &
                 *sqrt(del/alp)/(sig*sig)

      enddo
    enddo
  enddo

  return
end subroutine mdadmnmg


!***********************************************************************
!           GENERAL FUNCTIONS
!***********************************************************************
!--------------------------------------------------------------------
function ffjump(xx,bi,ee,aa,dd,bb)
!--------------------------------------------------------------------
  implicit none
  real(8) :: xx, bi, ee, aa, dd, bb, ffjump
!
  if( xx .le. aa-bb ) then
    ffjump=bi
   elseif( xx .ge. aa+bb ) then
    ffjump=ee
   else
    ffjump=1.0+tanh((xx-aa)/dd)/tanh(bb/dd)
    ffjump=bi+0.5*ffjump*(ee-bi)
   endif
!
   return
end function ffjump
!
