!***********************************************************************
!      RAISHIN code 3-dimensional Special Relativistic MHD version 
!      main program
!      written by Y. Mizuno
!***********************************************************************
!
program main

  use pram
  implicit none
  include 'mpif.h' ! for MPI
!
!======================================================================@
!    Difinition for variables
!======================================================================@
!
! uu: converved variables (cell-center)
! uo: 1 step before data of uu
! u0: initial value of uu
! uusr: store data for restart
! uh, us, un: 1st, 2nd, 3rd step value of uu
! uu*r, uu*l: conserved variables (cell-surface)
!
  real(8), allocatable :: uu(:,:,:,:), uo(:,:,:,:), &
          u0(:,:,:,:), uh(:,:,:,:), us(:,:,:,:), un(:,:,:,:), &
          uupr(:,:,:,:), uusr(:,:,:,:), uuir(:,:,:,:), uuil(:,:,:,:), &
          uujr(:,:,:,:), uujl(:,:,:,:), uukr(:,:,:,:), uukl(:,:,:,:)
!
! uri: primitive variables (cell-center)
! urio: 1 step before data of uri
! uri0: initial value of uri
! urisr: store data for restart
! uri*r, uri*l: premitive variables (cell-surface)
!
  real(8), allocatable :: uri(:,:,:,:), urio(:,:,:,:), &
          uri0(:,:,:,:), urip(:,:,:,:), urisr(:,:,:,:), &
          uriir(:,:,:,:), urijr(:,:,:,:), urikr(:,:,:,:), &
          uriil(:,:,:,:), urijl(:,:,:,:), urikl(:,:,:,:)
!
! Metric terms: hh,hi,ho: cell-center
!               h*i,h*j,h*k: cell-boundary
!
  real(8), allocatable :: hh(:,:,:,:), &
          hhi(:,:,:,:), hhj(:,:,:,:), hhk(:,:,:,:), &
          hi(:,:,:,:,:), hii(:,:,:,:), hij(:,:,:,:), hik(:,:,:,:), &
          ho(:,:,:,:,:), hoi(:,:,:,:), hoj(:,:,:,:), hok(:,:,:,:) 
  real(8), allocatable :: gm(:,:,:,:,:)
!
! ww: numerical flux (cell-center)
! wwo: 1step before numerical flux (cell-center) 
! ww*r, ww*l: numerical flux (cell-boundary)
!
  real(8), allocatable :: ww(:,:,:,:,:), wwo(:,:,:,:,:), &
          wwir(:,:,:,:), wwjr(:,:,:,:), wwkr(:,:,:,:), & 
          wwil(:,:,:,:), wwjl(:,:,:,:), wwkl(:,:,:,:)
!
! sf: source term
!
  real(8), allocatable :: sf(:,:,:,:)
!
! cmax, cmin: maximum and minimum wave speed
!
  real(8), allocatable :: cmaxi(:,:,:), cmini(:,:,:), &
          cmaxj(:,:,:), cminj(:,:,:), cmaxk(:,:,:), cmink(:,:,:)
!
! Primitive variables
!
  real(8), allocatable :: b1(:,:,:), b2(:,:,:), b3(:,:,:), &
          v1(:,:,:), v2(:,:,:), v3(:,:,:), de(:,:,:), pr(:,:,:)
!
! Grid: x1: cell-center, x1a: cell-boundary
!       dx1: x1(i+1)-x1(i), dx1b=x1a(i+1)-x1a(i)
!       akap1=dt/dx1, akap1b=dt/dx1b
!
  real(8) :: x1(imax), x2(jmax), x3(kmax), &
             x1a(imax), x2a(jmax), x3a(kmax), &
             dx1(imax), dx2(jmax), dx3(kmax), &
             dx1b(imax-1), dx2b(jmax-1), dx3b(kmax-1), &
             akap1(imax), akap2(jmax), akap3(kmax), &
             akap1b(imax-1), akap2b(jmax-1), akap3b(kmax-1)
!
!     for Kolmogorov-like Power-law Spectrum 
!
  real(8) :: wkn(nkmax), pkn1d(nkmax), pkn2d(nkmax), &
             pkn3d(nkmax), thetan(nkmax), phin(nkmax), &
             thetan1(nkmax), phin1(nkmax)
!
!    for GRMHD part
!
!
! Gravitational Potential
!
  real(8), allocatable :: pg(:,:,:), pgi(:,:,:), pgj(:,:,:), &
                          pgk(:,:,:), dpg(:,:,:)
!
! additional Metric terms: wok, sgk: cell-center
!               woki,wokj,wokk: cell-boundary
!
  real(8), allocatable :: wok(:,:,:,:), woki(:,:,:,:), wokj(:,:,:,:), &
                          wokk(:,:,:,:)
  real(8), allocatable :: sgk(:,:,:,:,:)
  real(8), allocatable :: tenr(:,:,:,:,:)
!
  character*256 :: filename, filename1
  integer :: i, j, k, m, n 
  integer :: is, ie, is1, ie1
  integer :: js, je, js1, je1
  integer :: ks, ke, ks1, ke1 
  integer :: iup, idown, jup, jdown, kup, kdown
  integer :: icputable(-1:iprocs,-1:jprocs,-1:kprocs)
  real(8) :: dt, dtshot, dt1, dt2, dt3, dtcfl, dtcflg 
  real(8) :: dx1a, dx2a, dx3a, akap1a, akap2a, akap3a
  real(8) :: time, timeh, tnext, timesr, timeo, time1
  integer :: istop, ierror
  integer :: it0, it, ihi, its, itsr, ihisr, isr, iit
  integer :: ig, nd
  real(8) :: cputime
  integer :: nm0, nm1, nm2, ider
  real(8) :: cl, amdamax, amdamin, totpkn1d, totpkn2d, totpkn3d
  real(8) :: e0, rj, vj, etaj, betaj, cmac, bm
  integer :: imagj
  real(8) :: vpre1, vplu1, opre1, omk1
  integer :: npe, myrank, merr, myranki, myrankj, myrankk
!
!======================================================================@
! for MPI
  call mpi_init(merr)
  call mpi_comm_size(mpi_comm_world,npe   ,merr)
  call mpi_comm_rank(mpi_comm_world,myrank,merr)
!
  if(npe .ne. iprocs*jprocs*kprocs) then
    if(myrank .eq. 0) then
      write(*,*) 'Error: for ncpu \= iprocs*kprocs'
    endif
    call mpi_finalize(merr)
    stop
  endif 
!
  call makecputable(myrank,icputable,myranki,myrankj,myrankk)
  call pararange(1,imax,iprocs,myranki,is,ie)
  call pararange(1,jmax,jprocs,myrankj,js,je)
  call pararange(1,kmax,kprocs,myrankk,ks,ke)
!      
!    Buffer grid number
!
  if(myrank .eq. 0) then
    write(*,*) 'npe,iprocs,kprocs=', npe, iprocs, jprocs, kprocs
  endif
  write(*,*) 'myrank,myranki,myrankj,myrankk=', myrank, myranki, &
             myrankj, myrankk

  nm0=1
  nm1=1
      
  if(irec .le. 3 .or. irec .eq. 11) then
    nm0=2
  elseif(irec .eq. 4 .or. irec .ge. 6 .and. irec .le. 10) then
    nm0=3
  elseif(irec .eq. 5) then
    nm0=4
  endif
!      
  if(ict .eq. 0) then
    nm1=nm0
  elseif(ict .eq. 1) then
    nm1=nm0+1
  elseif(ict .eq. 2) then
    nm1=nm0+2
  elseif(ict .eq. 3) then
    nm1=nm0+3
  endif
      
  if(irec .le. 3) then
    ider=0
    nm2=nm1
  elseif(irec .eq. 4 .or. irec .eq. 6 .or. irec .eq. 7 .or. irec .eq. 8) then
    ider=1
    nm2=nm1+1
  elseif(irec .eq. 5) then
    ider=2
    nm2=nm1+2
  endif
!
  if(myranki .eq. 0) then
    is1=1
  else
    is1=is-nm1
  endif
  if(myranki .eq. iprocs-1) then
    ie1=ie
  else
    ie1=ie+nm1
  endif
!
  if(myrankj .eq. 0) then
    js1=1
  else
    js1=js-nm1
  endif
  if(myrankj .eq. jprocs-1) then
    je1=je
  else
    je1=je+nm1
  endif
!
  if(myrankk .eq. 0) then
    ks1=1
  else
    ks1=ks-nm1
  endif
  if(myrankk .eq. kprocs-1) then
    ke1=ke
  else
    ke1=ke+nm1
  endif
!
  if(ie1-is1+1 .le. 2*nm1 .or. je1-js1+1 .le. 2*nm1 &
      .or. ke1-ks1+1 .le. 2*nm1) then
    write(*,*) 'Error: for grid size < minimum number of grid'
    write(*,*) 'myrank, ith grid #, jth grid #, kth grid # =',myrank, &
               ie1-is1+1, je1-js1+1, ke1-ks1+1, 2*nm1 
    call mpi_finalize(merr)
    stop        
  endif
!
  write(*,*) 'is,ie,js,je,ks,ke=',is,ie,js,je,ks,ke 
!
!======================================================================@
!  Allocate variables
!======================================================================@
!
  allocate( uu(nv,is1:ie1,js1:je1,ks1:ke1), &
            uo(nv,is1:ie1,js1:je1,ks1:ke1), &
            u0(nv,is1:ie1,js1:je1,ks1:ke1), &
            uh(nv,is1:ie1,js1:je1,ks1:ke1), &
            us(nv,is1:ie1,js1:je1,ks1:ke1), &
            un(nv,is1:ie1,js1:je1,ks1:ke1), &
            uupr(nv,is1:ie1,js1:je1,ks1:ke1), &
            uusr(nv,is1:ie1,js1:je1,ks1:ke1), &
            uuir(nv,is1:ie1,js1:je1,ks1:ke1), &
            uuil(nv,is1:ie1,js1:je1,ks1:ke1), &
            uujr(nv,is1:ie1,js1:je1,ks1:ke1), &
            uujl(nv,is1:ie1,js1:je1,ks1:ke1), &
            uukr(nv,is1:ie1,js1:je1,ks1:ke1), &
            uukl(nv,is1:ie1,js1:je1,ks1:ke1), stat=merr )

  allocate( uri(nv,is1:ie1,js1:je1,ks1:ke1), &
            urio(nv,is1:ie1,js1:je1,ks1:ke1), &
            uri0(nv,is1:ie1,js1:je1,ks1:ke1), &
            urip(nv,is1:ie1,js1:je1,ks1:ke1), &
            urisr(nv,is1:ie1,js1:je1,ks1:ke1), &
            uriir(nv,is1:ie1,js1:je1,ks1:ke1), &
            uriil(nv,is1:ie1,js1:je1,ks1:ke1), &
            urijr(nv,is1:ie1,js1:je1,ks1:ke1), &
            urijl(nv,is1:ie1,js1:je1,ks1:ke1), &
            urikr(nv,is1:ie1,js1:je1,ks1:ke1), &
            urikl(nv,is1:ie1,js1:je1,ks1:ke1), stat=merr )

  allocate( hh(0:3,is1:ie1,js1:je1,ks1:ke1), &
            hhi(0:3,is1:ie1,js1:je1,ks1:ke1), &
            hhj(0:3,is1:ie1,js1:je1,ks1:ke1), &
            hhk(0:3,is1:ie1,js1:je1,ks1:ke1), &
            hi(3,nv,is1:ie1,js1:je1,ks1:ke1), &
            hii(nv,is1:ie1,js1:je1,ks1:ke1), &
            hij(nv,is1:ie1,js1:je1,ks1:ke1), &
            hik(nv,is1:ie1,js1:je1,ks1:ke1), &
            ho(3,nv,is1:ie1,js1:je1,ks1:ke1), &
            hoi(nv,is1:ie1,js1:je1,ks1:ke1), &
            hoj(nv,is1:ie1,js1:je1,ks1:ke1), &
            hok(nv,is1:ie1,js1:je1,ks1:ke1), &
            gm(0:3,3,is1:ie1,js1:je1,ks1:ke1), stat=merr )

  allocate( ww(3,nv,is1:ie1,js1:je1,ks1:ke1), &
            wwo(3,nv,is1:ie1,js1:je1,ks1:ke1), &
            wwir(nv,is1:ie1,js1:je1,ks1:ke1), &
            wwil(nv,is1:ie1,js1:je1,ks1:ke1), &
            wwjr(nv,is1:ie1,js1:je1,ks1:ke1), &
            wwjl(nv,is1:ie1,js1:je1,ks1:ke1), &
            wwkr(nv,is1:ie1,js1:je1,ks1:ke1), &
            wwkl(nv,is1:ie1,js1:je1,ks1:ke1), stat=merr )

  allocate( sf(2:5,is1:ie1,js1:je1,ks1:ke1), stat=merr )

  allocate( cmaxi(is1:ie1,js1:je1,ks1:ke1), &
            cmini(is1:ie1,js1:je1,ks1:ke1), &
            cmaxj(is1:ie1,js1:je1,ks1:ke1), &
            cminj(is1:ie1,js1:je1,ks1:ke1), &
            cmaxk(is1:ie1,js1:je1,ks1:ke1), &
            cmink(is1:ie1,js1:je1,ks1:ke1), stat=merr )

  allocate( b1(is1:ie1,js1:je1,ks1:ke1), b2(is1:ie1,js1:je1,ks1:ke1), &
            b3(is1:ie1,js1:je1,ks1:ke1), &
            v1(is1:ie1,js1:je1,ks1:ke1), v2(is1:ie1,js1:je1,ks1:ke1), &
            v3(is1:ie1,js1:je1,ks1:ke1), &
            de(is1:ie1,js1:je1,ks1:ke1), pr(is1:ie1,js1:je1,ks1:ke1), &
            stat=merr )

  allocate( pg(is1:ie1,js1:je1,ks1:ke1), pgi(is1:ie1,js1:je1,ks1:ke1), &
            pgj(is1:ie1,js1:je1,ks1:ke1),pgk(is1:ie1,js1:je1,ks1:ke1), &
            dpg(is1:ie1,js1:je1,ks1:ke1), stat=merr )

  allocate( wok(3,is1:ie1,js1:je1,ks1:ke1), &
            woki(3,is1:ie1,js1:je1,ks1:ke1), &
            wokj(3,is1:ie1,js1:je1,ks1:ke1), &
            wokk(3,is1:ie1,js1:je1,ks1:ke1), &
            sgk(3,3,is1:ie1,js1:je1,ks1:ke1), &
            tenr(0:3,0:3,is1:ie1,js1:je1,ks1:ke1), stat=merr)

!
!======================================================================@
!    file open for restart
!======================================================================@
!
!    open( unit=1, file='ipram', status='old')
  write(filename1,991) myrank
  open( unit=9, file=filename1, status='unknown', &
        form='unformatted')
!
 991 format ('restart',i3.3,'.outdat')
!
!======================================================================@
!     Initialization of varibles
!======================================================================@
!
!     Fixed parameter
!
  istop=0
  ierror=0
!
  nd=0
!
  dt=0.01d0
  dtshot=tmax/float(nshot)
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1
        sf(2,i,j,k)=0.d0
        sf(3,i,j,k)=0.d0
        sf(4,i,j,k)=0.d0
        sf(5,i,j,k)=0.d0
      enddo
    enddo
  enddo
!
  do k=ks1,ke1
    do j=js1,je1
      do i=is1,ie1
        do m=1,nv
           ww(1,m,i,j,k)=0.d0
           ww(2,m,i,j,k)=0.d0
           ww(3,m,i,j,k)=0.d0
          
           uri(m,i,j,k)=0.d0
           urip(m,i,j,k)=0.d0
           uriir(m,i,j,k)=0.d0
           urijr(m,i,j,k)=0.d0
           urikr(m,i,j,k)=0.d0
           uriil(m,i,j,k)=0.d0
           urijl(m,i,j,k)=0.d0
           urikl(m,i,j,k)=0.d0
         
           uupr(m,i,j,k)=0.d0
           uuir(m,i,j,k)=0.d0
           uujr(m,i,j,k)=0.d0
           uukr(m,i,j,k)=0.d0
           uuil(m,i,j,k)=0.d0
           uujl(m,i,j,k)=0.d0
           uukl(m,i,j,k)=0.d0
         
           wwir(m,i,j,k)=0.d0
           wwjr(m,i,j,k)=0.d0
           wwkr(m,i,j,k)=0.d0
           wwil(m,i,j,k)=0.d0
           wwjl(m,i,j,k)=0.d0
           wwkl(m,i,j,k)=0.d0         
        enddo
      enddo
    enddo
  enddo
!
!======================================================================@
!      Print list of parameters
!======================================================================@
!
  if(myrank .eq. 0) then
    write(6,*) 'job name: ',jname,',  date: ',date
    write(6,*) 'Restart : ',icres
    write(6,*) '         List of Parameters '
    write(6,*) 'dtime =',dt,',  tmax =', tmax
    write(6,*) 'itmax =',itmax
    write(6,*) 'imax  =',imax,',  jmax =',jmax,',  kmax =',kmax
    write(6,*) 'npe =',npe
    write(6,*) 'xmin  =',xmin,',  xmax =', xmax
    write(6,*) 'ymin  =',ymin,',  ymax =', ymax
    write(6,*) 'zmin  =',zmin,',  zmax =', zmax
    write(6,*) 'model =',model,',  metric =',metric
    write(6,*) 'ieos =',ieos
    write(6,*) 'c0 =',c0
    write(6,*) 'gamma =',gam,',  gam0 =',gam0
    write(6,*) 'irec =',irec,', icha =',icha
    write(6,*) 'ihll =',ihll,',  ict = ',ict
    write(6,*) 'iwvec =',iwvec
  endif
!
!======================================================================@
!     Grid generation
!======================================================================@
!
  call grid(x1,x2,x3,dx1,dx2,dx3,dx1a,dx2a,dx3a, &
            dx1b,dx2b,dx3b,x1a,x2a,x3a)
!
!======================================================================@
!     Initial setting for Metric (coordinates)
!======================================================================@
!
!
!     Calculation of Gravitational Potential
!
  call gravp(pg,dpg,pgi,pgj,pgk,x1,x2,x3,x1a,x2a,x3a, &
             is1,ie1,js1,je1,ks1,ke1)
!
  call nrbhshr(wok,sgk,woki,wokj,wokk,is1,ie1,js1,je1,ks1,ke1)

  if( metric .eq. 1 ) then
    call carmet(hh,hhi,hhj,hhk,is1,ie1,js1,je1,ks1,ke1)
    call cargeo(gm,is1,ie1,js1,je1,ks1,ke1)
  elseif( metric .eq. 2 ) then
    call cylmet(hh,x1,hhi,hhj,hhk,x1a,is1,ie1,js1,je1,ks1,ke1)
    call cylgeo(gm,hh,is1,ie1,js1,je1,ks1,ke1)
  elseif( metric .eq. 3 ) then
    call sphmet(hh,x1,x3,hhi,hhj,hhk,x1a,x3a,is1,ie1,js1,je1,ks1,ke1)
    call sphgeo(gm,x3,hh,is1,ie1,js1,je1,ks1,ke1)
  elseif( metric .eq. 102) then
    call grcmet(hh,x1,hhi,hhj,hhk,x1a,pg,pgi,pgj,pgk,is1,ie1,js1,je1,ks1,ke1)
    call grcgeo(gm,x1,x3,hh,pg,dpg,is1,ie1,js1,je1,ks1,ke1)
  elseif( metric .eq. 103) then
    call grpmet(hh,x1,x3,hhi,hhj,hhk,x1a,x3a,pg,pgi,pgj,pgk, &
                is1,ie1,js1,je1,ks1,ke1)
    call grpgeo(gm,x1,x3,hh,pg,dpg,is1,ie1,js1,je1,ks1,ke1)
  elseif( metric .eq. 203) then
    call kermet(hh,x1,x3,hhi,hhj,hhk,x1a,x3a,is1,ie1,js1,je1,ks1,ke1)
    call kergeo(gm,x1,x3,hh,is1,ie1,js1,je1,ks1,ke1)
    call kershr(wok,sgk,x1,x3,hh,gm,hhi,hhj,hhk,x1a,x3a,woki,wokj,wokk, &
                is1,ie1,js1,je1,ks1,ke1)
  else
    stop
  endif
!
  call calhoi(hh,hi,ho,hhi,hhj,hhk,hii,hij,hik,hoi,hoj,hok, &
              is1,ie1,js1,je1,ks1,ke1)
!
!>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
!
!    Researrt simulation or Not
!
  if( icres .eq. 0 ) then
!
!    Initial Model Setup for GRMHD
! 
    if(model .eq. 0) then
      call bhmodel(de,pr,v1,v2,v3,b1,b2,b3,pg,hh,wok,x1,x2,x3,&
                   is1,ie1,js1,je1,ks1,ke1)
    endif
!
!      Initial Model Setup for RMHD
! 
    if(model .eq. 23) then
      call md2dblast(de,pr,v1,v2,v3,b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1)
    elseif(model .eq. 26) then
      call md2dmagadv(de,pr,v1,v2,v3,b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1)
    elseif(model .eq. 29) then
      call md3dmagadv(de,pr,v1,v2,v3,b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1)
    elseif(model .eq. 27) then
      call md2drotor(de,pr,v1,v2,v3,b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1)
    elseif(model .eq. 31) then
      call md2dkh(de,pr,v1,v2,v3,b1,b2,b3,x1,x2,x3,is1,ie1,js1,je1,ks1,ke1)
    elseif(model .eq. 50) then
      call mdshoctub(de,pr,v1,v2,v3,b1,b2,b3,is1,ie1,js1,je1,ks1,ke1)
    elseif(model .eq. 51) then
      call mdalfvenpro(de,pr,v1,v2,v3,b1,b2,b3,x1,is1,ie1,js1,je1,ks1,ke1) 
    else
    endif
!
!         Primitive Variables Set (uri)
!
    do k=ks1,ke1
      do j=js1,je1
        do i=is1,ie1
!
          uri(1,i,j,k)=de(i,j,k)
          uri(2,i,j,k)=v1(i,j,k)
          uri(3,i,j,k)=v2(i,j,k)
          uri(4,i,j,k)=v3(i,j,k)
          uri(5,i,j,k)=pr(i,j,k)
          uri(6,i,j,k)=0.d0
          uri(7,i,j,k)=b1(i,j,k)
          uri(8,i,j,k)=b2(i,j,k)
          uri(9,i,j,k)=b3(i,j,k)
!
        enddo
      enddo
    enddo
!
!         Conserved Variables Set (uu)
!
    call caluu1(uri,uu,x1,is1,ie1,js1,je1,ks1,ke1)
!
!     Transfer of variables as initial value
!
    call ident(uu,uo,is1,ie1,js1,je1,ks1,ke1)
    call ident(uu,u0,is1,ie1,js1,je1,ks1,ke1)
    call ident(uu,uusr,is1,ie1,js1,je1,ks1,ke1)
    call ident(uri,urio,is1,ie1,js1,je1,ks1,ke1)
    call ident(uri,uri0,is1,ie1,js1,je1,ks1,ke1)
    call ident(uri,urisr,is1,ie1,js1,je1,ks1,ke1)
!
!>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
!
!     Initialize for time advance loop start
!
    it0=0
    ihi=0
    time=0.d0
    timeh=0.d0
    tnext=0.d0
  endif
!
! come back point if icres .eq. 1
!
!======================================================================@
!     Startup for restart simulation
!======================================================================@
!
!  If restart is 'on' skip the initial setup and move this line
!
  if( icres .ne. 0 ) then
!    call restar(uu,u0,uri,uri0,it0,ihi,timeh,ks1,ke1)
    call restar2(uu,u0,uri,uri0,it0,ihi,timeh,nd,npe,myrank,&
                 is1,ie1,js1,je1,ks1,ke1)
!
!!      tnext=timeh+dtshot                               1997.12.04(Thu)
    tnext=(int(timeh/dtshot)+1)*dtshot
!
    call ident(uu,uusr,is1,ie1,js1,je1,ks1,ke1)
    call ident(uri,urisr,is1,ie1,js1,je1,ks1,ke1)
  endif
!
!**********************************************************************@
!     !! Time Advance Step Loop Start !!
!**********************************************************************@
! 
  its=(itmax-it0+1)/nshot
  if( its .lt. 1 ) then
    its = 1
  endif
  itsr=it0
  ihisr=ihi
  isr=0
  timesr=timeh
!
  do it=it0,itmax
!
!-----------------------------------------------------------------------
!     CPU Time Check                                1995.04.05
!-----------------------------------------------------------------------
!
!    call kclock(ig)
    if(myrank .eq. 0) then
      call cpu_time(cputime)
    endif
!
    call mpi_barrier(mpi_comm_world,merr)
    call mpi_bcast(cputime,1,mpi_double_precision,0,mpi_comm_world,merr)
!
!    if( ig .ge. icpu ) then
    if(cputime .ge. float(icpu)) then
      if(myrank .eq. 0) then
        write(6,*) ' CPU Time Over : cputime =',cputime,'(sec)'
       endif
       istop=1
    endif
!
    iit=it-it0
!
!x   if( mod(it,its) .eq. 0  .or. istop.eq.1 .or. it.eq.itmax ) then
!
!
!----------------------------------------------------------------------
!     Simulation time check
!----------------------------------------------------------------------
!
    if( dtmin.ge.0.d0 ) then
      timeo=time
      time=timeh
    else
      time = dt*float(it)
    endif
!
    if( time.gt.tmax ) then
      if(myrank .eq. 0) then
        write(6,*) '  '
        write(6,*) 'Congraturations! Run Succeeded! : time =' &
                    ,time,', tmax',tmax
        write(6,*) '  '
      endif
      istop = 1
    endif
!
!----------------------------------------------------------------------
!     Data store and output
!----------------------------------------------------------------------
!
    call mpi_barrier(mpi_comm_world,merr)
!
    if( time.ge.tnext .or. ierror.ne.0 &
        .or. istop.eq.1 .or. it.eq.itmax ) then
!
      if( ierror.eq.1 ) then
        write(6,*) 'stop by fault; time =',time
      endif
!
      ihi=ihi+it-it0+1
      if(time .eq. 0.0d0) then
       nd=0
      else
       nd=nd+1
      endif
!
! Data store for restart
!
      if( (isskip.eq.0 .or. istop.eq.1) .and. ierror.eq.0 ) then
        call store1(uu,u0,uri,uri0,it,ihi,time,nd,npe,myrank,&
                    is1,ie1,js1,je1,ks1,ke1)
      elseif( it.eq.itmax .and. ierror.eq.0 ) then
        call store1(uu,u0,uri,uri0,it,ihi,time,nd,npe,myrank,&
                    is1,ie1,js1,je1,ks1,ke1)
      elseif( mod(isr,isskip).eq.0 .and. ierror.eq.0 ) then
        call store1(uusr,u0,urisr,uri0,itsr,ihisr,timesr,nd,npe,myrank, &
                    is1,ie1,js1,je1,ks1,ke1)
        itsr=it
        ihisr=ihi
        timesr=time
        call ident(uu,uusr,is1,ie1,js1,je1,ks1,ke1)
        call ident(uri,urisr,is1,ie1,js1,je1,ks1,ke1)
      endif
!
! Data output as Simulation results
!
      if(myrank .eq. 0) then
        write(6,*) ' data output for analysis'
        write(6,*) 'time=',time
!        write(6,*) 'dtshot=',dtshot
!        write(6,*) 'tnext=',tnext
        write(6,*) 'cputime=',cputime
      endif
!
      call output1(uri,x1,x2,x3,time,nm1,nd,npe,myrank,myranki,myrankj,&
                   myrankk,is1,ie1,js1,je1,ks1,ke1)
!
      isr=isr+1
      tnext=tnext+dtshot
!
    endif
!
    if( ierror .ne. 0 .or. istop .eq. 1 .or. it .eq. itmax)  then
      goto 100
    endif
!
!=======================================================================
!      Start next time step calculation
!=======================================================================
!
!-----------------------------------------------------------------------
!     Variable tansfer (before 1st step)
!-----------------------------------------------------------------------
!
    call ident(uu,uo,is1,ie1,js1,je1,ks1,ke1)
    call ident(uri,urio,is1,ie1,js1,je1,ks1,ke1)
!
!!======================================================================@
!     Calcuration of dt for next time step
!======================================================================@
!
    if( dtmin.ge.0.d0 ) then

      call cdtcfl4(uri,hh,wok,dtcfl,dx1,dx2,dx3,dt1,dt2,dt3,nm1, &
                   is1,ie1,js1,je1,ks1,ke1)
!
! MPI
!
      call mpi_allreduce(dtcfl,dtcflg,1,mpi_double_precision,mpi_min, &
                         mpi_comm_world,merr)
      dt=cfl*dtcflg
!
      do i=1,imax-1
        akap1(i)=dt/dx1(i)
        akap1b(i)=dt/dx1b(i)
      enddo
      akap1a=dt/dx1a
!
      do j=1,jmax-1
        akap2(j)=dt/dx2(j)
        akap2b(j)=dt/dx2b(j)
      enddo
      akap2a=dt/dx2a
!
      do k=1,kmax-1
        akap3(k)=dt/dx3(k)
        akap3b(k)=dt/dx3b(k)
      enddo
      akap3a=dt/dx3a
!
      timeh=time+dt
    else
      timeh= dt*(float(it)+1.0)
    endif
       
    if( dt.lt.dtmin ) then
!      write(6,*) ' dt < dtmin: dt =',dt,', dtmin =',dtmin
      istop = 1
    endif
!
!-----------------------------------------------------------------------
!
    if(myrank .eq. 0) then
      write(6,620) 'Simulation running: it = ',it, &
                 ',  dt =',dt,', time=',time
620   format(1h ,a27,i6, 2(a7,1pe11.3))
    endif
!
!**********************************************************************@
!     First Step
!**********************************************************************@
!
!     Reconstraction Step
!
    call rec2(uri,x1,x2,x3,dx1,dx2,dx3,dx1b,dx2b,dx3b, &
              uriir,urijr,urikr,uriil,urijl,urikl, &
              uuir,uujr,uukr,uuil,uujl,uukl, &
              wwir,wwjr,wwkr,wwil,wwjl,wwkl,nm0,is1,ie1,js1,je1,ks1,ke1)
!
    if(metric .eq. 203) then
      call kaddwr2(wwir,wwjr,wwkr,wwil,wwjl,wwkl,&
                   uuir,uujr,uukr,uuil,uujl,uukl,&
                   woki,wokj,wokk,is1,ie1,js1,je1,ks1,ke1)
    endif
!
!     Calculation of Characteristics
!
    call calcha4a(uriir,urijr,urikr,uriil,urijl,urikl, &
                  cmaxi,cmini,cmaxj,cminj,cmaxk,cmink, &
                  hhi,hhj,hhk,woki,wokj,wokk,nm0, &
                  is1,ie1,js1,je1,ks1,ke1)
!
!     Calculation of Numerical flux 
!
    call hll(ww,x1,x2,x3,uri,uriir,uriil,urijr,urijl,urikr,urikl, &
             uuir,uujr,uukr,uuil,uujl,uukl, &
             wwir,wwjr,wwkr,wwil,wwjl,wwkl, &
             cmaxi,cmini,cmaxj,cminj,cmaxk,cmink,nm0, &
             is1,ie1,js1,je1,ks1,ke1)
!
    call calwwo(uu,wwo,uri,is1,ie1,js1,je1,ks1,ke1)
!
!      Constrained Transport
!
    call ct(ww,wwo,uri,nm0,nm1,is1,ie1,js1,je1,ks1,ke1)
!
!    call deriv(ww,ider,nm1,nm2,is1,ie1,js1,je1,ks1,ke1)
!
!     Calcuration of Source term
!
    if(metric .eq. 203) then
      call caltenr(tenr,uu,uri,wwo,wok,is1,ie1,js1,je1,ks1,ke1)
      call kaddsf1(sf,tenr,gm,wok,sgk,dt,is1,ie1,js1,je1,ks1,ke1)
    else
      call calsf(sf,uu,wwo,gm,dt,is1,ie1,js1,je1,ks1,ke1)
    endif
!
!     Time advance of first step
!
!y   call ident(uu,uh,is1,ie1,js1,je1,ks1,ke1)
!
    call rk2fst(uh,ww,hi,ho,uu,hii,hij,hik,hoi,hoj,hok,nm1, &
                akap1b,akap2b,akap3b,is1,ie1,js1,je1,ks1,ke1)
!
!     Add Source term
!
    call rk2adsff(uh,sf,nm1,is1,ie1,js1,je1,ks1,ke1)
!
!     Boundary condition for conserved variables
!
!y   call bnd4(uh,uo,x1,nm1,is1,ie1,js1,je1,ks1,ke1, &
!y             myranki,myrankj,myrankk,icputable)
!
!     Recovery step
!
    call recov(uh,uri,x1,x2,x3,nm1,is1,ie1,js1,je1,ks1,ke1)
!
!     Boundary condition for primitive variables
!
    call bnd4(uri,urio,x1,nm1,is1,ie1,js1,je1,ks1,ke1, &
              myranki,myrankj,myrankk,icputable)
!
!     Calculation of conserved variables at boundary region
!

    call calconv(uh,uri,x1,nm1,is1,ie1,js1,je1,ks1,ke1, &
                 myranki,myrankj,myrankk)
!
!      MPI data exchange
!
    call mpi_barrier(mpi_comm_world,merr)
    call mpiex(uri,nm1,is1,ie1,js1,je1,ks1,ke1,myranki,myrankj,&
               myrankk,icputable)
    call mpiex(uh,nm1,is1,ie1,js1,je1,ks1,ke1,myranki,myrankj,&
               myrankk,icputable)
!
!     Check and Correction (velocity)
!
    if( adampa.ne.0.d0 ) then
      call spdlia1(uri,uh,istop,ierror,is1,ie1,js1,je1,ks1,ke1)
    endif
!
!     Check and Correction (pressure and density)
!
    call pminmax1a(uh,uri,is1,ie1,js1,je1,ks1,ke1)
!
!     Artificial Damping (primitive variables) 
!
    if(model .ne. 22) then
      if(adamp.gt.0.d0 .and. rdamp.gt.0.d0) then
        call damp4(uh,uri,uri0,x1,x3,is1,ie1,js1,je1,ks1,ke1)
      endif
    elseif(model .eq. 22) then
      if(adamp.gt.0.d0) then
        call damp4a(uh,uri,uri0,x1,x3,is1,ie1,js1,je1,ks1,ke1)
      endif
    endif
!
!     Transfer of variables
!
    call ident(uh,uu,is1,ie1,js1,je1,ks1,ke1)
!
!**********************************************************************@
!     Second Step
!**********************************************************************@
!
!     Reconstraction Step
!
    call rec2(uri,x1,x2,x3,dx1,dx2,dx3,dx1b,dx2b,dx3b, &
              uriir,urijr,urikr,uriil,urijl,urikl, &
              uuir,uujr,uukr,uuil,uujl,uukl, &
              wwir,wwjr,wwkr,wwil,wwjl,wwkl,nm0,is1,ie1,js1,je1,ks1,ke1)
!
    if(metric .eq. 203) then
      call kaddwr2(wwir,wwjr,wwkr,wwil,wwjl,wwkl,&
                   uuir,uujr,uukr,uuil,uujl,uukl,&
                   woki,wokj,wokk,is1,ie1,js1,je1,ks1,ke1)
    endif
!
!     Calculation of Characteristics
!
    call calcha4a(uriir,urijr,urikr,uriil,urijl,urikl, &
                  cmaxi,cmini,cmaxj,cminj,cmaxk,cmink, &
                  hhi,hhj,hhk,woki,wokj,wokk,nm0, &
                  is1,ie1,js1,je1,ks1,ke1)
!
!     Calculation of Numerical flux 
!
    call hll(ww,x1,x2,x3,uri,uriir,uriil,urijr,urijl,urikr,urikl, &
             uuir,uujr,uukr,uuil,uujl,uukl, &
             wwir,wwjr,wwkr,wwil,wwjl,wwkl, &
             cmaxi,cmini,cmaxj,cminj,cmaxk,cmink,nm0, &
             is1,ie1,js1,je1,ks1,ke1)

    call calwwo(uu,wwo,uri,is1,ie1,js1,je1,ks1,ke1)
!
!       Constrained Transport
!
    call ct(ww,wwo,uri,nm0,nm1,is1,ie1,js1,je1,ks1,ke1)
!
!    call deriv(ww,ider,nm1,nm2,is1,ie1,js1,je1,ks1,ke1)
!
!     Calcuration of source term 
!
    if(metric .eq. 203) then
      call caltenr(tenr,uu,uri,wwo,wok,is1,ie1,js1,je1,ks1,ke1)
      call kaddsf1(sf,tenr,gm,wok,sgk,dt,is1,ie1,js1,je1,ks1,ke1)
    else
      call calsf(sf,uu,wwo,gm,dt,is1,ie1,js1,je1,ks1,ke1)
    endif
!
!     Time advance of second step
!
    if(irkt .eq. 2) then
      call rk2snd(us,ww,hi,ho,uo,uh,hii,hij,hik,hoi,hoj,hok,nm1, &
                  akap1b,akap2b,akap3b,is1,ie1,js1,je1,ks1,ke1)
    elseif(irkt .eq. 3) then
      call rk3snd(us,ww,hi,ho,uo,uh,hii,hij,hik,hoi,hoj,hok,nm1, &
                  akap1b,akap2b,akap3b,is1,ie1,js1,je1,ks1,ke1)
    endif
!
!     Add the Source Term
!
    if(irkt .eq. 2) then
      call rk2adsfs(us,sf,nm1,is1,ie1,js1,je1,ks1,ke1)
    elseif(irkt .eq. 3) then
      call rk3adsfs(us,sf,nm1,is1,ie1,js1,je1,ks1,ke1)
    endif
!
!     Boundary Condition for conserved variables
!
!y   call bnd4(us,uo,x1,nm1,is1,ie1,js1,je1,ks1,ke1,&
!y             myranki,myrankj,myrankk,icputable)
!
!     Recovery step
!
    call recov(us,uri,x1,x2,x3,nm1,is1,ie1,js1,je1,ks1,ke1)
!
!     Boundary Condition for primitive variables
!
    call bnd4(uri,urio,x1,nm1,is1,ie1,js1,je1,ks1,ke1,&
              myranki,myrankj,myrankk,icputable)
!
!     Calculation of conserved variables at boundary region
!
    call calconv(us,uri,x1,nm1,is1,ie1,js1,je1,ks1,ke1, &
                 myranki,myrankj,myrankk)
!
!      MPI data exchange
!
    call mpi_barrier(mpi_comm_world,merr)
    call mpiex(uri,nm1,is1,ie1,js1,je1,ks1,ke1,myranki,myrankj,&
               myrankk,icputable)
    call mpiex(us,nm1,is1,ie1,js1,je1,ks1,ke1,myranki,myrankj,&
               myrankk,icputable)
!
!     Check and Correction (velocity)
!
    if( adampa.ne.0.d0 ) then
      call spdlia1(uri,us,istop,ierror,is1,ie1,js1,je1,ks1,ke1)
    endif
!
!     Check and Correction (pressure and density)
!
    call pminmax1a(us,uri,is1,ie1,js1,je1,ks1,ke1)
!
!     Artificial Damping (primitive variables) 
!
    if(model .ne. 22) then
      if(adamp.gt.0.d0 .and. rdamp.gt.0.d0) then
        call damp4(us,uri,uri0,x1,x3,is1,ie1,js1,je1,ks1,ke1)
      endif
    elseif(model .eq. 22) then
      if(adamp.gt.0.d0) then
        call damp4a(us,uri,uri0,x1,x3,is1,ie1,js1,je1,ks1,ke1)
      endif
    endif
!
!     Transfer of variables
!
    call ident(us,uu,is1,ie1,js1,je1,ks1,ke1)
!
!**********************************************************************@
!     Additional calculation for third Step of time evolution
!**********************************************************************@
!ccc
    if(irkt .eq. 3) then
!ccc
!
!     Reconstraction Step
!
      call rec2(uri,x1,x2,x3,dx1,dx2,dx3,dx1b,dx2b,dx3b, &
                uriir,urijr,urikr,uriil,urijl,urikl, &
                uuir,uujr,uukr,uuil,uujl,uukl, &
                wwir,wwjr,wwkr,wwil,wwjl,wwkl,nm0,is1,ie1,js1,je1,ks1,ke1)
!
      if(metric .eq. 203) then
        call kaddwr2(wwir,wwjr,wwkr,wwil,wwjl,wwkl,&
                     uuir,uujr,uukr,uuil,uujl,uukl,&
                     woki,wokj,wokk,is1,ie1,js1,je1,ks1,ke1)
      endif
!
!     Calculation of Characteristics
!
      call calcha4a(uriir,urijr,urikr,uriil,urijl,urikl, &
                    cmaxi,cmini,cmaxj,cminj,cmaxk,cmink, &
                    hhi,hhj,hhk,woki,wokj,wokk,nm0, &
                    is1,ie1,js1,je1,ks1,ke1)
!
!     Calculation of Numerical flux 
!
      call hll(ww,x1,x2,x3,uri,uriir,uriil,urijr,urijl,urikr,urikl, &
               uuir,uujr,uukr,uuil,uujl,uukl, &
               wwir,wwjr,wwkr,wwil,wwjl,wwkl, &
               cmaxi,cmini,cmaxj,cminj,cmaxk,cmink,nm0, &
               is1,ie1,js1,je1,ks1,ke1)
!
      call calwwo(uu,wwo,uri,is1,ie1,js1,je1,ks1,ke1)
!
!      Constrained Transport
!
      call ct(ww,wwo,uri,nm0,nm1,is1,ie1,js1,je1,ks1,ke1)
!
!      call deriv(ww,ider,nm1,nm2,is1,ie1,js1,je1,ks1,ke1)
!
!     Calcuration of source term 
!
      if(metric .eq. 203) then
        call caltenr(tenr,uu,uri,wwo,wok,is1,ie1,js1,je1,ks1,ke1)
        call kaddsf1(sf,tenr,gm,wok,sgk,dt,is1,ie1,js1,je1,ks1,ke1)
      else
        call calsf(sf,uu,wwo,gm,dt,is1,ie1,js1,je1,ks1,ke1)
      endif
!
!     Time advance of second step
!
      call rk3trd(un,ww,hi,ho,uo,us,hii,hij,hik,hoi,hoj,hok,nm1, &
                  akap1b,akap2b,akap3b,is1,ie1,js1,je1,ks1,ke1)
!
!     Add the Source Term
!
      call rk3adsft(un,sf,nm1,is1,ie1,js1,je1,ks1,ke1)
!
!     Boundary Condition for conserved variables
!
!y     call bnd4(un,uo,x1,nm1,is1,ie1,js1,je1,ks1,ke1,&
!y               myranki,myrankj,myrankk,icputable)
!
!     Recovery Step
!
      call recov(un,uri,x1,x2,x3,nm1,is1,ie1,js1,je1,ks1,ke1)
!
!     Boundary Condition for primitive variables
!
      call bnd4(uri,urio,x1,nm1,is1,ie1,js1,je1,ks1,ke1, &
                myranki,myrankj,myrankk,icputable)
!
!     Calculation of conserved variables at boundary region
!
      call calconv(un,uri,x1,nm1,is1,ie1,js1,je1,ks1,ke1, &
                   myranki,myrankj,myrankk)
!
!     MPI data exchange
!
      call mpi_barrier(mpi_comm_world,merr)
      call mpiex(uri,nm1,is1,ie1,js1,je1,ks1,ke1,myranki,myrankj,&
                 myrankk,icputable)
      call mpiex(un,nm1,is1,ie1,js1,je1,ks1,ke1,myranki,myrankj,&
                 myrankk,icputable)
!
!     Check and Correction of velocity
!
      if( adampa.ne.0.d0 ) then
        call spdlia1(uri,un,istop,ierror,is1,ie1,js1,je1,ks1,ke1)
      endif
!
!     Check and Correction of pressure
!
      call pminmax1a(un,uri,is1,ie1,js1,je1,ks1,ke1)
!
!     Artificial Damping (primitive variables) 
!
      if(model .ne. 22) then
        if(adamp.gt.0.d0 .and. rdamp.gt.0.d0) then
          call damp4(un,uri,uri0,x1,x3,is1,ie1,js1,je1,ks1,ke1)
        endif
      elseif(model .eq. 22) then
        if(adamp.gt.0.d0) then
          call damp4a(un,uri,uri0,x1,x3,is1,ie1,js1,je1,ks1,ke1)
        endif
      endif
!
!     Transfer of variables
!
      call ident(un,uu,is1,ie1,js1,je1,ks1,ke1)
!ccc
    endif
!
  enddo
!
!**********************************************************************@
!     !! Time Step Loop End !!
!**********************************************************************@
!
100 continue

  call mpi_finalize(merr)

  if(myrank.eq. 0) then
    write(6,*) '== end =='
  endif

  rewind(8)
!
!======================================================================@
!     CPU Time for End of Job
!======================================================================@
!
!y  it0=it
!y  call kclock(ig)
!y  write(6,*) 'CPU Time at End of Job :',ig,'sec'
!   .
!======================================================================@
  stop
end program main
