!
!     Transonic Solutions for Initial Condition
!
!-----------------------------------------------------------------------
subroutine  caxpoi(xxp,fxp,h0)
!-----------------------------------------------------------------------
  use pram, only : gam, c0  
  implicit none

  real(8) :: xxp, fxp, h0, hh, gg, ddd, ffx, dfx, as, al, gl, uu
  integer :: ierr
!
!-------------------------
!     X-point Searching
!-------------------------
!
!     xxp = rxp/rbh ( rxp is never used in this subroutine !)
!
  ierr=0
!
  hh=h0
!
!     Parameter Calulation
!
  gg=(3.0*gam-2.0)/4.0/(gam-1.0)
!
!     Initial Value Setting
!
  if( hh .gt. 1.d0 .and. hh .lt. 1.1d0) then
    if( gam .gt. 1.666d0 .and. gam .lt. 1.667d0 ) then
      ddd=(gg*hh)**2+3.0*(gg**2+0.75**2*3.0-3.0*gg)
      xxp=2.0*gg*hh**2-9.0/4.0+hh*sqrt(ddd)
      xxp=xxp/3.0/(hh**2-1.0)
      xxp=2.0*xxp
    else
      xxp=(8.*gg-9.)/(4.*(hh-1.))
!     <=====>  xxp = 0
    endif
  else
    xxp=5.d0
  endif
!
  call newton1(xxp,ffx,dfx,gg,hh,ierr)
!
  if( ierr .ne. 0 ) then
    write(6,*) 'Error with newton in caxpoi: ierr =',ierr
    write(6,*) 'Stop at caxpoi'
    stop
  endif
!
  as=1.0-1.0/xxp
  al=sqrt(as)
  gl=hh*((as-1.0)/(1.0+3.0*as)+gam-1.0)/((gam-1.0)*al)
  uu=sqrt(gl**2-1.0)
  fxp=(hh/(al*gl)-1.0)*(al*xxp**2*uu)**(gam-1.0)
!
  return
end subroutine caxpoi
!
!-----------------------------------------------------------------------
subroutine washk1(x1,u,xxp,fxp,h0)
!-----------------------------------------------------------------------
!     unit caution:  x* = r(x1*) / rbh
!-------------------------
!     Down Wind State
!-------------------------

  use pram, only: gam, c0, imax, rbh
  implicit none

  integer :: i, merr
  real(8) :: x1(imax), u(imax)
  real(8) :: xxp, fxp, fdw, dfd, fxd, h0, hh, xx, al, uu
  integer :: ierr, ierro
!
  fxd=fxp
  hh=h0
!
  ierr=0
!
  do i=1,imax
    xx=x1(i)/rbh
!
    al=sqrt(1.0-1.0/xx)
!
    if( hh .gt. 0.d0 .and. xx .gt. 0.d0 ) then
      uu=(al*fxd/hh)**(1.0/(gam-1.0))/(al*xx**2)
    else
      write(6,*) 'error: initial value of uu'
      write(6,*) ' hh =',hh,'  xx =',xx
      write(6,*) ' stop at down streame'
      stop
    endif
!
    if( xx .lt. xxp ) then
      uu=sqrt((hh/al)**2-1.0)
    endif
      
!
    ierro=ierr
    call newton2(uu,xx,fdw,dfd,fxd,al,hh,ierr)
    ierr=ierro+ierr
!
    if( ierr .ne. 0 ) then
      write(6,*) 'Error with newton in washk1: ierr =',ierr
      write(6,*) 'xx =', xx, i
      write(6,*) 'Stop at washk1'
      stop
    endif
!
    u(i)=uu
!
  enddo
!
  return
end subroutine washk1
!
!-----------------------------------------------------------------------
subroutine washk2(x1,uw,xxp,fxp,h0)
!-----------------------------------------------------------------------
!     unit caution:  x* = r(x1)* / rbh
!-------------------------
!     Up Wind State
!-------------------------
!
  use pram, only: gam, c0, imax, rbh
  implicit none
!
  integer :: i
  real(8) :: x1(imax), uw(imax)
  real(8) :: xxp, fxp, fdw, dfd, fxd, h0, hh, xx, al, uu
  integer :: ierr, ierro
!
  fxd=fxp
  hh=h0
!
  ierr=0
!
  do i=1,imax
    xx=x1(i)/rbh
!
    al=sqrt(1.0-1.0/xx)
!
    if( hh .gt. 0.d0 .and. xx .gt. 0.d0 ) then
      uu=sqrt((hh/al)**2-1.0)
!       uu=(al*fxd/hh)**(1.0/(gam-1.0))/(al*xx**2)
    else
      write(6,*) 'error: initial value of uu'
      write(6,*) ' hh =',hh,'  xx =',xx
      write(6,*) ' stop at down streame'
      stop
    endif
!
    ierro=ierr
    call newton2(uu,xx,fdw,dfd,fxd,al,hh,ierr)
    ierr=ierro+ierr
!                                         
    if( ierr .ne. 0 ) then
      write(6,*) 'Error with newton in washk2: ierr =',ierr
      write(6,*) 'xx =',xx, i
      write(6,*) 'Stop at washk2'
      stop
    endif
!
    uw(i)=uu
!
  enddo
!
  return
end subroutine washk2
!
!-----------------------------------------------------------------------
subroutine  cupwin(ff,rr,amon,h0)
!-----------------------------------------------------------------------
  use pram, only: gam, c0, imax, rbh
  implicit none

  real(8) :: ff, rr, amon, h0, hh, xx, amonc, al, uu, gl, fuwf, duwf
  integer :: ierr
!
!  external fuwf,duwf
!
  ierr=0
!
  hh=h0
  xx=rr/rbh
  amonc=amon
!
!     Parameter Calulation
!
!     Initial Value Setting
!
  al=sqrt(1.0-rbh/rr)
  uu=sqrt((hh/al)**2-1.0)
!
  call newton3(uu,fuwf,duwf,hh,al,amonc,ierr)
!                                                
  if( ierr .ne. 0 ) then
    write(6,*) 'Error with newton in cupwin: ierr =',ierr
    write(6,*) 'Stop at cupwin'
    stop
  endif
!
  gl=sqrt(uu**2+1.0)
  ff=(hh/(al*gl)-1.0)*(al*xx**2*uu)**(gam-1.0)

  return
end subroutine cupwin
!
!-----------------------------------------------------------------------
subroutine newton1(xx,ffx,dfx,gg,hh,ierr)
!-----------------------------------------------------------------------
  use pram, only : gam, c0, iter  
  implicit none

  real(8) :: xx, ffx, dfx, gg, hh, xo, huge, ermax
  integer :: i, ierr, ite
!
  huge =1.0d5 
  ierr=0
  ite=0
  ermax=1.0d-5
!
  do i=1,iter
    xo=xx
    ffx= (hh**2-1.)*xx**3+(2.25-2.*gg*hh**2)*xx**2 &
       +((hh*gg)**2-27./16.)*xx+0.75**3
    dfx= 3.*(hh**2-1.)*xx**2+(4.5-4.*gg*hh**2)*xx+(hh*gg)**2-27./16.

    xx = xx - ffx/dfx

    if( abs(xx-xo) .lt. ermax ) then
    
    elseif( abs(xx-xo) .gt. huge ) then
      ierr=0
      write(6,*) 'Error in newton: at iteration =',i
      write(6,*) ' xx - xo diverge!: xx - xo =',xx-xo
      write(6,*) ' will be stop by ierr = 1 signal'
    else
      xx=xx
      ite=i
    endif
  enddo
!
  if( ite .ge. iter ) then
    ierr=1
  endif

  return
end subroutine newton1
!
!-----------------------------------------------------------------------
subroutine newton2(uu,xx,fdw,dfd,fxd,al,hh,ierr)
!-----------------------------------------------------------------------
  use pram, only : gam, c0, iter  
  implicit none

  real(8) :: uu, xx, fdw, dfd, fxd,al, hh, uo, huge, ermax, gl
  integer :: i, ierr, ite
!
  huge =1.0d5 
  ierr=0
  ite=0
  ermax=1.0d-5
!
  do i=1,iter
    uo=uu
    gl=sqrt(uu**2+1.0)
!
    fdw=(al*xx**2)**(gam-1.0)*(hh/al/gl-1.0)*uu**(gam-1.0)
    fdw=fdw-fxd
!    
    dfd=(al*xx**2)**(gam-1.0)*uu**(gam-2.0)/gl**3
    dfd=dfd*(hh/al*(-uu**2+(gam-1.0)*gl**2)-(gam-1.0)*gl**3)

    uu = uu - fdw/dfd

    if( abs(uu-uo) .lt. ermax ) then
    
    elseif( abs(uu-uo) .gt. huge ) then
      ierr=0
      write(6,*) 'Error in newton: at iteration =',i
      write(6,*) ' xx - xo diverge!: xx - xo =',uu-uo
      write(6,*) ' will be stop by ierr = 1 signal'
    else
      uu=uu
      ite=i
    endif
  enddo
!
  if( ite .ge. iter ) then
    ierr=1
  endif

  return
end subroutine newton2
!
!-----------------------------------------------------------------------
subroutine newton3(uu,fuwf,duwf,hh,al,amonc,ierr)
!-----------------------------------------------------------------------
  use pram, only : gam, c0, iter  
  implicit none

  real(8) :: uu, fuwf, duwf, gl, hh, al, amonc,uo, huge, ermax
  integer :: i, ierr, ite
!
  huge =1.0d5 
  ierr=0
  ite=0
  ermax=1.0d-5
!
  do i=1,iter
    uo=uu
    gl=sqrt(uu**2+1.0)
    fuwf=(hh/al*uu**2/gl + (gam-1.0)/gam*(hh/al/gl-1.0))/uu-amonc
    duwf=(gam-1.0+hh/al*((2.0-gam)*uu**2-gam+1.0)/gl**1.5)/gam/uu**2
!
    uu = uu - fuwf/duwf

    if( abs(uu-uo) .lt. ermax ) then
    
    elseif( abs(uu-uo) .gt. huge ) then
      ierr=0
      write(6,*) 'Error in newton: at iteration =',i
      write(6,*) ' xx - xo diverge!: xx - xo =',uu-uo
      write(6,*) ' will be stop by ierr = 1 signal'
    else
      uu=uu
      ite=i
    endif
  enddo
!
  if( ite .ge. iter ) then
    ierr=1
  endif

  return
end subroutine newton3
!
!-----------------------------------------------------------------------
function  fuwi(uu,hh,al)
!-----------------------------------------------------------------------
  use pram, only : gam, c0
  implicit none

  real(8) :: uu, fuwi, al, hh
!
  fuwi=(al*(gam-1.0)/hh)**2*(uu**2+1.0)**3 &
      - ((gam-2.0)*uu**2+gam-1.0)**2
!
  return
end function fuwi
!-----------------------------------------------------------------------
function  dfu(uu,hh,al)
!-----------------------------------------------------------------------
  use pram, only : gam, c0
  implicit none

  real(8) :: uu, dfu, al, hh
!
  dfu=uu*( 6.0*(al*(gam-1.0)/hh)**2*(uu**2+1.0)**2 &
     -  4.0*(gam-2.0)*((gam-2.0)*uu**2+gam-1.0))
!
  return
end function dfu
!
!-----------------------------------------------------------------------
subroutine calws1(rr,ur,te,cnst1,cnst2)
!----------------------------------------------------------------------- 
  use pram, only : gam, c0, rbh, iter
  implicit none

  integer :: nnn
  real(8) :: rr, ur, te, cnst1, cnst2, gam1
  real(8) :: f1, g1, dfx, dfy, dgx, dgy, det, dx1, dy1, delta, deltaend
  real(8) :: ursq, rrsq, tmp3a, tmp3b, xx_old, yy_old 

!     Parameter for Bondi Accretion

  delta=1.d0
  deltaend=1.d-3
  gam1=1.d0/(gam-1.d0)

  do nnn=0,iter
    rrsq=rr*rr
    ursq=ur*ur

    f1=ur*(te**gam1)*rrsq-cnst1
    tmp3a=(1.+(1.+gam1)*te)**2
    tmp3b=1.-(rbh/rr)+ursq
    g1=tmp3a*tmp3b-cnst2
      
    dfx=(te**gam1)*rrsq
    dfy=ur*gam1*(te**(gam1-1.))*rrsq
    dgx=2.*tmp3a*ur
    dgy=2.*(1.+gam1)*(1.+(1.+gam1)*te)*tmp3b

    det=dfx*dgy-dfy*dgx

!      write(*,*) 'f1, g1=', f1, g1
!      write(*,*) 'dfx,dfy,dgx,dgy=', dfx, dfy, dgx, dgy

    if(det .eq. 0.d0) then
      write(6,*) ' >> Jacobian =0 in grbondi'
      write(6,*) ' >> delta, iter =', delta, nnn,', at x1:',rr
 !       stop
    endif

    if(det .eq. 0.d0) then
      dx1=0.d0
      dy1=0.d0
      delta=0.d0
    else        
      dx1=(-dgy*f1+dfy*g1)/det
      dy1=(dgx*f1-dfx*g1)/det

      delta=abs((dx1+dy1)/2.0)
    endif

    if(delta .lt. deltaend) then
      goto 101
    endif

    xx_old=ur
    yy_old=te

    ur=ur+dx1
    te=te+dy1

  enddo

  if(delta .gt. deltaend) then 
    write(6,*) ' >> Not convergence in grbondi'
    write(6,*) ' >> delta, iter =', delta, nnn,', at x1:',rr
    stop
  endif
!
!----------------------------------------------------
!     Calculation of primitive variables
!----------------------------------------------------
!
  101 continue 

  return
end subroutine calws1
